/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import java.sql.SQLException;
import java.util.List;

/**
 * Objects of this class contain zero or more {@link ChildEntity} objects.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public interface EntitiesContainer<T extends ChildEntity> {

    /**
     * Returns a specific child of this container, selecting it by name and
     * filtering it by a set of possible status: if the child doesn't have one
     * of the status specified, this method returns null.
     * <p>
     * <strong>If no status is specified it is assumed that all status are
     * valid.</strong>
     *
     * @param childName the name of the {@link ChildEntity} to retrieve.
     * @param status a set of allowed {@link Status} for the {@link ChildEntity}
     * to retrieve.
     * @return a {@link ChildEntity} or null if there is no child matching the
     * specified criteria.
     */
    T getChild(String childName, Status... status);

    /**
     * Adds a {@link ChildEntity} specifying its name. If the child entity has
     * never been read from the database it is loaded when this method is
     * called. If the entity was marked as removed, it will marked as added
     * again.
     */
    T addChild(String childName) throws SQLException;

    /**
     * Removes a {@link ChildEntity} given its name.
     * <p>
     * The child isn't really removed but only marked as removed (the removal
     * will happen when the {@link TapSchema#save()} method will be called).
     *
     * @param childName the name of the {@code ChildEntity} to remove.
     */
    void removeChild(String childName);

    /**
     * Retrieves a list of names of child entities that can be added.
     * <p>
     * The list contains only names of entities that have never been added:
     * entities the instances of which don't exist yet or entities having
     * {@link Status} {@code LOADED}.
     */
    List<String> getAddableChildrenNames();

    /**
     * Tells if a {@link ChildEntity} can be added to this container, given its
     * name.
     *
     * @param childName the name of the entity to add.
     * @return true if the {@code ChildEntity} can be added, false otherwise.
     */
    boolean isAddable(String childName);

    /**
     * Retrieves a list of the children of this container filtering them by a
     * set of possible status.
     * <p>
     * <strong>If no status is specified it is assumed that all status are
     * valid.</strong>
     */
    List<T> getChildren(Status... statuses);

    /**
     * Retrieves all children of this container having
     * {@link Status} {@code ADDED_PERSISTED} or {@code ADDED_NOT_PERSISTED}.
     */
    List<T> getAddedChildren();

    /**
     * Retrieves all children of this container having
     * {@link Status} {@code ADDED_PERSISTED}, {@code ADDED_NOT_PERSISTED},
     * {@code TO_REMOVE} or {@code REMOVED_NOT_PERSISTED}.
     */
    List<T> getAddedOrRemovedChildren();
}
