/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import it.inaf.ia2.tsm.model.ColumnModel;
import java.io.Serializable;

/**
 * Store the value of an entity property (that corresponds to a column of the
 * mapped table).
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class EntityProperty<T> implements Serializable {

    private static final long serialVersionUID = 6735553751078589085L;

    private ColumnModel propertyModel;
    private Class<T> type;
    private T defaultValue;
    private T originalValue;
    private T value;
    private boolean changed;

    private EntityProperty() {
    }

    public EntityProperty(ColumnModel propertyModel, T defaultValue) {
        this.propertyModel = propertyModel;
        this.type = propertyModel.getJavaType();
        this.defaultValue = defaultValue;
        this.init(defaultValue);
    }

    /**
     * Retrieve the current value.
     */
    public T getValue() {
        return value;
    }

    public T getOriginalValue() {
        return originalValue;
    }

    public T getDefaultValue() {
        return defaultValue;
    }

    public <X> void setValue(X value) {
        this.setValue(value, true);
    }

    private <X> void setValue(X value, boolean checkUpdatable) {
        assert value == null || value.getClass() == type;
        if (checkUpdatable && !propertyModel.isUpdatable()) {
            throw new UnsupportedOperationException("This EntityProperty instance (" + propertyModel.getName() + ") is not updatable");
        }
        if (value == null && !propertyModel.isNullable()) {
            throw new IllegalArgumentException("This EntityProperty instance (" + propertyModel.getName() + ") doesn't allow null values");
        }
        this.value = (T) value;
    }

    public boolean isChanged() {
        if (changed) {
            return true;
        }
        if (originalValue == null) {
            return value != null;
        }
        return !originalValue.equals(value);
    }

    public void setChanged(boolean changed) {
        this.changed = changed;
    }

    /**
     * Initialize the value.
     */
    public final <X> void init(X initialValue) {
        if (!propertyModel.isNullable() && initialValue == null) {
            initialValue = (X) propertyModel.getDefaultValue();
        }
        setValue(initialValue, false);
        this.originalValue = (T) initialValue;
    }

    public void save() {
        this.originalValue = value;
    }

    public Class getType() {
        return type;
    }

    public boolean isUpdatable() {
        return propertyModel.isUpdatable();
    }

    public boolean isNullable() {
        return propertyModel.isNullable();
    }
}
