/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import it.inaf.ia2.tsm.model.ColumnModel;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * DataModel for consistency checking result.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class ConsistencyChecks implements Serializable {

    private static final long serialVersionUID = 4412404312756740093L;
    private final static Logger LOG = LoggerFactory.getLogger(ConsistencyChecks.class);

    private final List<InconsistentColumnProperty> inconsistencies;
    private final Set<String> unexisingSchemas;
    private final Set<String> unexisingTables;
    private final Set<ColumnHolder> unexistingColumns;
    private final Set<KeyHolder> unexistingKeys;
    private final Map<String, Set<String>> missingTables;
    private final Map<String, Set<String>> tablesToAdd;
    private final Map<ColumnHolder, ColumnModel> missingColumns;
    private final Set<ColumnHolder> columnsToAdd;
    private final List<WrongDataType> wrongDataTypes;
    private boolean missingObscore;
    private boolean obscoreToAdd;

    // This will not consider an inconsistency: it is only used to display a warning
    private final Set<ColumnHolder> unaddedOptionalColumns;

    public ConsistencyChecks() {
        inconsistencies = new ArrayList<>();
        unexisingSchemas = new HashSet<>();
        unexisingTables = new HashSet<>();
        unexistingColumns = new HashSet<>();
        unexistingKeys = new HashSet<>();
        missingTables = new HashMap<>();
        tablesToAdd = new HashMap<>();
        missingColumns = new HashMap<>();
        columnsToAdd = new HashSet<>();
        unaddedOptionalColumns = new HashSet<>();
        wrongDataTypes = new ArrayList<>();
    }

    public void addInconsistency(InconsistentColumnProperty problemDescription) {
        inconsistencies.add(problemDescription);
    }

    public List<InconsistentColumnProperty> getInconsistencies() {
        return inconsistencies;
    }

    public Set<String> getUnexisingSchemas() {
        return unexisingSchemas;
    }

    public void addUnexistingSchema(String schemaName) {
        unexisingSchemas.add(schemaName);
    }

    public Set<String> getUnexisingTables() {
        return unexisingTables;
    }

    public void addUnexistingTable(String schemaName, String tableSimpleName) {
        unexisingTables.add(schemaName + "." + tableSimpleName);
    }

    public Set<ColumnHolder> getUnexisingColumns() {
        return unexistingColumns;
    }

    public void addUnexistingColumn(String schemaName, String tableName, String columnName) {
        unexistingColumns.add(new ColumnHolder(schemaName, tableName, columnName));
    }

    public void addUnexistingKey(String keyId, String fromTable, String[] fromColumns, String targetTable, String[] targetColumns) {
        if (keyId == null) {
            throw new IllegalArgumentException("key_id can't be null");
        }
        unexistingKeys.add(new KeyHolder(keyId, fromTable, fromColumns, targetTable, targetColumns));
    }

    public Set<KeyHolder> getUnexistingKeys() {
        return unexistingKeys;
    }

    public void addMissingColumn(ColumnHolder columnHolder, ColumnModel columnModel) {
        // Removing table from unexisting columns set
        unexistingColumns.remove(columnHolder);
        missingColumns.put(columnHolder, columnModel);
    }

    public Map<ColumnHolder, ColumnModel> getMissingColumns() {
        return missingColumns;
    }

    private void addTableToMap(String schemaName, String tableName, Map<String, Set<String>> map) {
        Set<String> tables = map.get(schemaName);
        if (tables == null) {
            tables = new HashSet<>();
            map.put(schemaName, tables);
        }
        tables.add(tableName);
    }

    public void addMissingTable(String schemaName, String tableName) {
        // Removing table from unexisting table set
        unexisingTables.remove(String.format("%s.%s", schemaName, tableName));

        // Removing table from unexisting columns set
        Iterator<ColumnHolder> ite = unexistingColumns.iterator();
        while (ite.hasNext()) {
            ColumnHolder ch = ite.next();
            if (ch.getSchemaName().equals(schemaName) && ch.getTableName().equals(tableName)) {
                ite.remove();
            }
        }

        addTableToMap(schemaName, tableName, missingTables);
    }

    public void addTableToAdd(String schemaName, String tableName) {
        addTableToMap(schemaName, tableName, tablesToAdd);
    }

    public Map<String, Set<String>> getTablesToAdd() {
        return tablesToAdd;
    }

    public Map<String, Set<String>> getMissingTables() {
        return missingTables;
    }

    public void addColumnToAdd(ColumnHolder columnHolder) {
        columnsToAdd.add(columnHolder);
    }

    public Set<ColumnHolder> getColumnsToAdd() {
        return columnsToAdd;
    }

    public boolean isMissingObscore() {
        return missingObscore;
    }

    public void setMissingObscore(boolean missingObscore) {
        this.missingObscore = missingObscore;
    }

    public boolean isObscoreToAdd() {
        return obscoreToAdd;
    }

    public void setObscoreToAdd(boolean obscoreToAdd) {
        this.obscoreToAdd = obscoreToAdd;
    }

    public void addUnaddedOptionalColumn(ColumnHolder columnHolder) {
        unaddedOptionalColumns.add(columnHolder);
    }

    public Set<ColumnHolder> getUnaddedOptionalColumns() {
        return unaddedOptionalColumns;
    }

    public void addWrongDataType(ColumnHolder columnHolder, String wrongDataType, String correctDataType, String adqlCorrectDataType, Integer size) {
        // If datatype needs to be changed inconsistency on it doesn't make sense anymore.
        Iterator<InconsistentColumnProperty> ite = inconsistencies.iterator();
        while (ite.hasNext()) {
            InconsistentColumnProperty inconsistency = ite.next();
            if (inconsistency.getColumnHolder().equals(columnHolder)
                    && (inconsistency.getKey().equals(Column.DATATYPE_KEY)
                    || inconsistency.getKey().equals(Column.SIZE_KEY)
                    || inconsistency.getKey().equals(Column.ARRAYSIZE_KEY))) {
                ite.remove();
            }
        }

        WrongDataType wdt = new WrongDataType(columnHolder, wrongDataType, correctDataType, adqlCorrectDataType, size);
        wrongDataTypes.add(wdt);
    }

    public List<WrongDataType> getWrongDataTypes() {
        return wrongDataTypes;
    }

    public boolean isInconsistent() {
        return !inconsistencies.isEmpty() || !wrongDataTypes.isEmpty()
                || !unexisingSchemas.isEmpty() || !unexisingTables.isEmpty() || !unexistingColumns.isEmpty()
                || !missingTables.isEmpty() || !missingColumns.isEmpty()
                || !columnsToAdd.isEmpty() || !tablesToAdd.isEmpty()
                || obscoreToAdd || missingObscore;
    }

    public boolean isHasWarnings() {
        return !unaddedOptionalColumns.isEmpty();
    }
}
