/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.model;

import java.util.ArrayList;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Performs inheritance between TASMAN schema definition XML files.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class XMLMerger {

    private static final Logger LOG = LoggerFactory.getLogger(XMLMerger.class);

    private final Document parentDoc;
    private final Document childDoc;

    /**
     * @param parentDoc the parent XML schema definition
     * @param childDoc the XML schema definition which inherits from parent
     */
    public XMLMerger(Document parentDoc, Document childDoc) throws ParserConfigurationException {
        // Cloning parent document
        this.parentDoc = cloneDocument(parentDoc);
        this.childDoc = childDoc;
    }

    private Document cloneDocument(Document document) throws ParserConfigurationException {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = dbf.newDocumentBuilder();
        Document clonedDocument = builder.newDocument();
        Node originalRoot = document.getDocumentElement();
        Node clonedRoot = clonedDocument.importNode(originalRoot, true);
        clonedDocument.appendChild(clonedRoot);
        return clonedDocument;
    }

    /**
     * Returns an XML document which is the result of the document inheritance.
     */
    public Document getMergedDocument() {

        Element root = parentDoc.getDocumentElement();
        Element inheritingRoot = childDoc.getDocumentElement();

        // Adding attributes for root node
        addAttributes(root, inheritingRoot);
        // visit recursively
        visitAndModifyTree(root, inheritingRoot);

        return parentDoc;
    }

    private List<Element> getChildElements(Element parent) {
        List<Element> elements = new ArrayList<>();
        NodeList nodes = parent.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Node node = nodes.item(i);
            if (node instanceof Element) {
                elements.add((Element) node);
            }
        }
        return elements;
    }

    private void visitAndModifyTree(Element inheritedParentEl, Element inheritingParentEl) {
        if (inheritedParentEl != null) {
            for (Element inheritingChildEl : getChildElements(inheritingParentEl)) {
                Element inheritedElement = checkElement(inheritedParentEl, inheritingChildEl);
                visitAndModifyTree(inheritedElement, inheritingChildEl);
            }
        }
    }

    private Element checkElement(Element inheritedParent, Element child) {

        Element inheritedElement = null;

        String name = child.getAttribute("name");
        if (name != null && !name.isEmpty()) {
            inheritedElement = findElementByNameAttribute(inheritedParent, name);
        }

        if (child.getNodeName().equals("remove")) {
            if (inheritedElement != null) {
                // Remove node by name
                inheritedParent.removeChild(inheritedElement);
            } else {
                LOG.warn("Unable to remove element {}: no such element in parent", name);
            }
        } else {
            if (inheritedElement == null) {
                // Add new node
                Node importedChild = parentDoc.importNode(child, true);
                inheritedParent.appendChild(importedChild);
            } else {
                // Apply inheritance (override node)
                applyElementInheritance(inheritedElement, child);
            }
        }

        return inheritedElement;
    }

    private Element findElementByNameAttribute(Element parent, String name) {
        for (Element element : getChildElements(parent)) {
            if (name.equals(element.getAttribute("name"))) {
                return element;
            }
        }
        return null;
    }

    private void addAttributes(Element inheritedElement, Element inheritingElement) {
        NamedNodeMap attrs = inheritingElement.getAttributes();
        for (int i = 0; i < attrs.getLength(); i++) {
            Node node = attrs.item(i);
            if (node instanceof Attr) {
                Attr attr = (Attr) node;
                inheritedElement.setAttribute(attr.getName(), attr.getValue());
            }
        }
    }

    private void applyElementInheritance(Element inheritedElement, Element inheritingElement) {

        addAttributes(inheritedElement, inheritingElement);

        for (Element child : getChildElements(inheritingElement)) {
            List<Element> subChildren = getChildElements(child);
            // apply element-to-element override only in leaf nodes
            if (subChildren.isEmpty()) {
                Element childToReplace = findElement(inheritedElement, child);
                Node importedChild = parentDoc.importNode(child, true);
                if (childToReplace == null) {
                    if (!child.getNodeName().equals("remove")) {
                        inheritedElement.appendChild(importedChild);
                    }
                } else {
                    inheritedElement.replaceChild(importedChild, childToReplace);
                }
            }
        }
    }

    private Element findElement(Element parent, Element childToFind) {
        for (Element element : getChildElements(parent)) {
            if (element.getNodeName().equals(childToFind.getNodeName())) {
                return element;
            }
        }
        return null;
    }
}
