/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.datalayer;

import it.inaf.ia2.tsm.ColumnHolder;
import it.inaf.ia2.tsm.ConsistencyChecks;
import it.inaf.ia2.tsm.Key;
import it.inaf.ia2.tsm.TapSchema;
import it.inaf.ia2.tsm.TapSchemaEntity;
import it.inaf.ia2.tsm.model.ColumnModel;
import it.inaf.ia2.tsm.model.TableModel;
import it.inaf.ia2.tsm.model.SchemaModel;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This interface can be seen as a DAO (Data Access Object pattern), however it
 * has been called more generically a "broker to the database", because it
 * doesn't map exactly the CRUD operations, indeed it contains some specific
 * methods for interacting to the databases (executing SQL statement) and
 * retrieving/storing data regarding TAP SCHEMA. Generic operations are
 * implemented inside the abstract class {@link DBBrokerTemplate}. The latter is
 * then extended by specific MySQL and Postgres Implementations.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public interface DBBroker {

    /**
     * Returns the name of all schemata contained into the source database.
     */
    List<String> getAllSchemaNames() throws SQLException;

    /**
     * Returns the name of all schemata which structure is compatible with a
     * TAP_SCHEMA schema structure.
     *
     * @param allSchemas a list of all schemata names to use for searching;
     * names must be real schemata names (not renamed).
     */
    List<String> getAllTAPSchemaNames(List<String> allSchemas) throws SQLException;

    /**
     * Try to guess the version of an existing TAP_SCHEMA analyzing its shallow
     * structure (e.<!-- -->g.<!-- --> the presence of the set of columns
     * defined into the related XML model).
     *
     * @param tapSchemaName the real TAP_SCHEMA name (not the renamed one).
     * @return the name of the guessed TAP_SCHEMA version.
     */
    String detectVersion(String tapSchemaName) throws SQLException;

    /**
     * Returns a list of all the schemata exposed by an existing TAP_SCHEMA.
     * Queries the {@code TAP_SCHEMA.schemas} schema.
     *
     * @param tapSchemaName the real TAP_SCHEMA name (not the renamed one).
     */
    List<String> getExposedSchemas(String tapSchemaName) throws SQLException;

    /**
     * Returns all the tables owned by a schema, given its name.
     *
     * @param schemaName the real schema name (not the renamed one).
     */
    List<String> getAllTablesNames(String schemaName) throws SQLException;

    /**
     * Returns the table type ("table" or "view") for all the tables owned by a
     * schema, given its name.
     *
     * @param schemaName the real schema name (not the renamed one).
     * @return a {@code Map} having table names as keys and table types as
     * values.
     */
    Map<String, String> getAllTableTypes(String schemaName) throws SQLException;

    /**
     * Returns the list of the names of the columns contained in a given table.
     *
     * @param schemaName the real schema name (not the renamed one).
     * @param tableName the name of the table.
     */
    List<String> getAllColumnsNames(String schemaName, String tableName) throws SQLException;

    /**
     * Returns the metadata of all the columns contained in a given table.
     *
     * @param schemaName the real schema name (not the renamed one).
     * @param tableSimpleName the name of the table.
     * @param tableModel the {@link TableModel} of the owner column if it is a
     * TAP_SCHEMA table or the ObsCore table, null otherwise. This is used for
     * consistency checking on these tables datatype.
     * @param dataTypeMode the {@link DataTypeMode} used in the selected
     * TAP_SCHEMA version; this is used only if the previous parameter is not
     * null.
     * @return a {@code Map} having the column names as keys and the column
     * metadata as values.
     */
    Map<String, Map<String, Object>> getAllColumnsMetadata(String schemaName, String tableSimpleName, TableModel tableModel, DataTypeMode dataTypeMode) throws SQLException;

    /**
     * Returns a list of all the foreign key relationships the columns of which
     * are contained in a given schema.
     *
     * @param schemaName the schema name as exposed by the TAP_SCHEMA
     * (eventually renamed).
     * @param realSchemaName the real schema name (not renamed).
     */
    List<Key> getKeys(TapSchema tapSchema, String schemaName, String realSchemaName) throws SQLException;

    /**
     * Retrieves TAP_SCHEMA items saved into an existing TAP_SCHEMA, for a given
     * TAP_SCHEMA table, given its {@link TableModel}. For example, if the
     * {@code TableModel} models the {@code columns} table, this method will
     * return a list representing all the rows in the {@code columns} table.
     *
     * @param tapSchemaName the real TAP_SCHEMA name (not the renamed one).
     * @param tableModel the model for the TAP_SCHEMA table to read.
     * @return A {@code List} representing all the rows in a specific TAP_SCHEMA
     * table. Each element of the list is a {@code Map} having column names as
     * keys and row values as values.
     */
    List<Map<String, Object>> getSavedItems(String tapSchemaName, TableModel tableModel) throws SQLException;

    /**
     * Inserts a new row in a TAP_SCHEMA table.
     *
     * @param tapSchemaName the real TAP_SCHEMA name (not the renamed one).
     * @param entity the entity to insert (this is used to specify the table to
     * use and the values to insert).
     * @param conn the database connection.
     */
    void insertItem(String tapSchemaName, TapSchemaEntity entity, Connection conn) throws SQLException;

    /**
     * Updates an existing row in a TAP_SCHEMA table.
     *
     * @param tapSchemaName the real TAP_SCHEMA name (not the renamed one).
     * @param entity the entity to update (this is used to specify the table to
     * use and the new values to insert).
     * @param conn the database connection.
     * @param whereCondition a SQL {@code WHERE} condition to add to the query.
     * @param whereParams a sequence of parameters for the {@code WHERE}
     * condition.
     */
    void updateItem(String tapSchemaName, TapSchemaEntity entity, Connection conn, String whereCondition, Object... whereParams) throws SQLException;

    /**
     * Creates all the tables of a TAP_SCHEMA.
     *
     * @param tapSchemaName the real TAP_SCHEMA name (not the renamed one).
     * @param tapSchemaModel the {@link SchemaModel} used to represent the
     * TAP_SCHEMA schema.
     */
    void createTapSchemaStructure(String tapSchemaName, SchemaModel tapSchemaModel) throws SQLException;

    /**
     * Creates all the tables of an ivoa schema.
     *
     * @param ivoaSchemaModel the {@link SchemaModel} used to represent the ivoa
     * schema.
     * @param realIvoaSchemaName the real name of the ivoa schema (not the
     * renamed one).
     */
    void createIvoaSchemaStructure(SchemaModel ivoaSchemaModel, String realIvoaSchemaName) throws SQLException;

    /**
     * Saves all the TAP_SCHEMA modifications into the database (creates a new
     * TAP_SCHEMA or update an existing one).
     */
    void save(TapSchema tapSchema) throws SQLException;

    /**
     * Create a new table into the database.
     *
     * @param schemaName the real schema name (not the renamed one).
     * @param tableModel the {@link TableModel} representing the table to
     * create.
     */
    void createTable(String schemaName, TableModel tableModel) throws SQLException;

    /**
     * Alter an existing table adding a new column.
     *
     * @param columnHolder a model representing a reference to a column which
     * doesn't exist yet.
     * @param columnModel the model used for representing the column structure.
     */
    void addColumn(ColumnHolder columnHolder, ColumnModel columnModel) throws SQLException;

    /**
     * Retrieves all the keys where the {@code from_table} or
     * {@code target_table} properties start by a given parameter, which must be
     * a schema name or a complete table name ({@code schema_name} plus
     * {@code table_name}). In this way it is possible to select keys that needs
     * to be removed both if the consistency checking mechanism has detected an
     * inexistent schema or an inexistent table.
     *
     * @param tapSchemaName the real TAP_SCHEMA name (not the renamed one).
     * @param like a string representing a schema name or a table complete name
     * ({@code schema_name} plus {@code table_name}).
     *
     * @see it.inaf.ia2.tsm.ConsistencyChecks
     */
    Set<String> getKeysToRemove(String tapSchemaName, String like) throws SQLException;

    /**
     * Deletes rows from the TAP_SCHEMA.
     *
     * @param keysToRemoveIds a {@code Set} of all the identifiers of the keys
     * to remove; the logic used to create this set is inside the
     * {@link it.inaf.ia2.tsm.TapSchemaMender} class.
     *
     * @see it.inaf.ia2.tsm.ConsistencyChecks
     * @see it.inaf.ia2.tsm.TapSchemaMender
     */
    void deleteUnexistingEntities(String tapSchemaName, ConsistencyChecks consistencyChecks, Set<String> keysToRemoveIds) throws SQLException;

    /**
     * Retrieves all the key identifiers related to an inexistent column.
     *
     * @param tapSchemaName the real TAP_SCHEMA name (not the renamed one).
     * @param unexistingColumn a reference to the inexistent column.
     * @return a {@code Set} of key identifier
     *
     * @see it.inaf.ia2.tsm.ConsistencyChecks
     * @see it.inaf.ia2.tsm.TapSchemaMender
     */
    Set<String> getKeysToRemoveFromUnexistingColumn(String tapSchemaName, ColumnHolder unexistingColumn) throws SQLException;

    /**
     * Update the (wrong) property value of column exposed by a TAP_SCHEMA.
     *
     * @param tapSchemaName the real TAP_SCHEMA name (not the renamed one).
     * @param completeTableName the complete name of the table
     * ({@code schema_name} plus {@code table_name}).
     * @param columnName the name of the column.
     * @param key the name of the TAP_SCHEMA column property.
     * @param value the new value.
     *
     * @see it.inaf.ia2.tsm.ConsistencyChecks
     * @see it.inaf.ia2.tsm.TapSchemaMender
     */
    void updateTapSchemaColumnValue(String tapSchemaName, String completeTableName, String columnName, String key, Object value) throws SQLException;

    /**
     * Alter an existing TAP_SCHEMA or ObsCore column having an incoherent
     * datatype according to its {@link ColumnModel}.
     *
     * @see it.inaf.ia2.tsm.ConsistencyChecks
     * @see it.inaf.ia2.tsm.TapSchemaMender
     */
    void alterDataType(String schemaName, String tableName, String columnName, String adqlDataType, Integer size) throws SQLException;
}
