/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import it.inaf.ia2.tsm.model.XMLModelsLoader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.junit.Test;
import static org.junit.Assert.*;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class TestXMLMerging {

    public String getXMLString(Document doc) throws TransformerException {
        TransformerFactory tFactory = TransformerFactory.newInstance();
        Transformer transformer = tFactory.newTransformer();
        transformer.setOutputProperty(OutputKeys.INDENT, "yes");

        DOMSource source = new DOMSource(doc);
        StreamResult result = new StreamResult(new StringWriter());
        transformer.transform(source, result);
        return result.getWriter().toString();
    }

    
    @Test
    public void test() throws Exception {

        String[] files = new String[]{"doc0.xml", "doc1.xml", "doc2.xml"};

        XMLModelsLoader loader = new XMLModelsLoader(files);

        loader.load(TestXMLMerging.class.getClassLoader());

        Map<String, Document> documents = loader.getDocuments();

        Element schv0 = documents.get("0").getDocumentElement();
        Element schv1 = documents.get("1").getDocumentElement();
        Element schv2 = documents.get("2").getDocumentElement();

        System.out.println("------------- 000 -------------");
        System.out.println(getXMLString(documents.get("0")));
        System.out.println("-------------------------------");
        System.out.println("------------- 111 -------------");
        System.out.println(getXMLString(documents.get("1")));
        System.out.println("-------------------------------");
        System.out.println("------------- 222 -------------");
        System.out.println(getXMLString(documents.get("2")));
        System.out.println("-------------------------------");
        
        assertEquals("0", schv0.getAttribute("version"));
        assertEquals("1", schv1.getAttribute("version"));
        assertEquals("2", schv2.getAttribute("version"));

        // Checking final merge:
        List<Element> tables = getElements(schv2.getElementsByTagName("table"));
        Element tab0 = getElementByAttributeName(tables, "tab0");
        Element tab1 = getElementByAttributeName(tables, "tab1");
        Element tab2 = getElementByAttributeName(tables, "tab2");
        Element tab3 = getElementByAttributeName(tables, "tab3");
        Element tab4 = getElementByAttributeName(tables, "tab4");

        // Checking first level table removal
        assertNull(tab0);
        // Checking second level table removal
        assertNull(tab1);

        assertNotNull(tab2);

        // Checking first level table add
        assertNotNull(tab3);
        // Checking second level table add
        assertNotNull(tab4);

        // Checking column override (first and second level)
        List<Element> tab2Columns = getElements(tab2.getElementsByTagName("column"));
        Element col2 = getElementByAttributeName(tab2Columns, "col2");
        assertEquals("CHAR", getTextElement(col2, "type"));
        assertEquals("true", getTextElement(col2, "mandatory"));
        assertEquals("100", getTextElement(col2, "size"));
        assertEquals("false", getTextElement(col2, "updatable"));
    }

    private String getTextElement(Element parent, String childName) {
        return parent.getElementsByTagName(childName).item(0).getTextContent().trim();
    }

    private List<Element> getElements(NodeList nodes) {
        List<Element> elements = new ArrayList<>();
        for (int i = 0; i < nodes.getLength(); i++) {
            Node node = nodes.item(i);
            if (node instanceof Element) {
                elements.add((Element) node);
            }
        }
        return elements;
    }

    private Element getElementByAttributeName(List<Element> elements, String name) {
        for (Element element : elements) {
            if (name.equals(element.getAttribute("name"))) {
                return element;
            }
        }
        return null;
    }
}
