/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import it.inaf.ia2.tsm.datalayer.DBWrapper;
import it.inaf.ia2.tsm.datalayer.DatabaseType;
import it.inaf.ia2.tsm.datalayer.Credentials;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import static org.junit.Assert.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class TestAll {

    private static final Logger LOG = LoggerFactory.getLogger(TestAll.class);

    private static final int SCHEMAS_COUNT = 2; // minimum value: 2
    private static final int TABLES_COUNT = 3;

    private static List<DBWrapper> dbWrappers;
    private static TapSchemaSettings settings;

    public TestAll() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {

        dbWrappers = new ArrayList<>();

        // MYSQL
        Credentials mysqlCredentials = EmbeddedDatabases.mariadbCredentials();

        // POSTGRES
        Credentials postgresCredentials = EmbeddedDatabases.postgresCredentials();

        DBWrapper dbWrapper = new DBWrapper(mysqlCredentials);
        dbWrapper.testConnections();
        dbWrappers.add(dbWrapper);

        dbWrapper = new DBWrapper(postgresCredentials);
        dbWrapper.testConnections();
        dbWrappers.add(dbWrapper);

        settings = new TapSchemaSettings();
        settings.setTapSchemaName("test_tap_schema");
        settings.setTapSchemaVersion("1.1-IA2");
        settings.setHasObscore(true);
        settings.setIvoaSchemaName("test_ivoa");
        settings.setObscoreVersion("1.1");

        // Mix!
        //dbWrappers.add(new DBWrapper(mysqlCredentials, postgresCredentials));
        //dbWrappers.add(new DBWrapper(postgresCredentials, mysqlCredentials));
    }

    @AfterClass
    public static void tearDownClass() {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    private void removeTestingDatabases() throws SQLException {
        for (DBWrapper dbWrapper : dbWrappers) {
            try (Connection sourceConnection = dbWrapper.getSourceConnection();
                    Statement statement = sourceConnection.createStatement()) {
                DatabaseType dbType = dbWrapper.getSourceDatabaseType();

                // Removing keys between schema1 and schema0
                if (null == dbType) {
                    throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                } else {
                    switch (dbType) {
                        case MYSQL:
                            try (ResultSet rs = statement.executeQuery("SHOW DATABASES WHERE `Database` = 'sch1'")) {
                                // The constraint can be removed only if sch1 exists
                                if (rs.next()) {
                                    statement.executeUpdate("ALTER TABLE sch1.table0 DROP FOREIGN KEY sch0table0id_constraint");
                                }
                            }
                            break;
                        case POSTGRES:
                            try (ResultSet rs = statement.executeQuery("SELECT schema_name FROM information_schema.schemata WHERE schema_name = 'sch1'")) {
                                // The constraint can be removed only if sch1 exists
                                if (rs.next()) {
                                    statement.executeUpdate("ALTER TABLE sch1.table0 DROP CONSTRAINT sch0table0id_constraint");
                                }
                            }
                            break;
                        default:
                            throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                    }

                    for (int i = 0; i < SCHEMAS_COUNT; i++) {
                        switch (dbType) {
                            case MYSQL:
                                statement.executeUpdate("DROP DATABASE IF EXISTS sch" + i);
                                break;
                            case POSTGRES:
                                statement.executeUpdate("DROP SCHEMA IF EXISTS sch" + i + " CASCADE");
                                break;
                            default:
                                throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                        }
                    }
                }
                try (Connection tapSchemaConnection = dbWrapper.getTapSchemaConnection();
                        Statement stat = tapSchemaConnection.createStatement()) {
                    dbType = dbWrapper.getTapSchemaDatabaseType();
                    if (null == dbType) {
                        throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                    } else {
                        switch (dbType) {
                            case MYSQL:
                                stat.executeUpdate("DROP DATABASE IF EXISTS test_tap_schema");
                                stat.executeUpdate("DROP DATABASE IF EXISTS test_ivoa");
                                break;
                            case POSTGRES:
                                stat.executeUpdate("DROP SCHEMA IF EXISTS test_tap_schema CASCADE");
                                stat.executeUpdate("DROP SCHEMA IF EXISTS test_ivoa CASCADE");
                                break;
                            default:
                                throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                        }
                    }
                }
            }
        }
    }

    private void setUpTestingDatabases() throws SQLException {
        removeTestingDatabases();

        for (DBWrapper dbWrapper : dbWrappers) {

            DatabaseType dbType = dbWrapper.getSourceDatabaseType();

            try (Connection connection = dbWrapper.getSourceConnection()) {

                try (Statement statement = connection.createStatement()) {

                    if (null == dbType) {
                        throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                    } else {
                        switch (dbType) {
                            case MYSQL:
                                for (int i = 0; i < SCHEMAS_COUNT; i++) {
                                    statement.executeUpdate("CREATE DATABASE sch" + i);
                                    for (int j = 0; j < TABLES_COUNT; j++) {
                                        statement.executeUpdate("CREATE TABLE sch" + i + ".table" + j + " (\n"
                                                + "id INT PRIMARY KEY AUTO_INCREMENT,\n"
                                                + "value1 VARCHAR(255),\n"
                                                + "value2 FLOAT\n"
                                                + ");");
                                    }
                                }
                                break;
                            case POSTGRES:
                                for (int i = 0; i < SCHEMAS_COUNT; i++) {
                                    statement.executeUpdate("CREATE SCHEMA sch" + i);
                                    for (int j = 0; j < TABLES_COUNT; j++) {
                                        statement.executeUpdate("CREATE TABLE sch" + i + ".table" + j + " (\n"
                                                + "id BIGSERIAL PRIMARY KEY,\n"
                                                + "value1 VARCHAR(255),\n"
                                                + "value2 FLOAT\n"
                                                + ");");
                                    }
                                }
                                break;
                            default:
                                throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                        }
                    }
                }

                LOG.info("dbs created");

                try (Statement statement = connection.createStatement()) {
                    for (int j = 0; j < TABLES_COUNT - 1; j++) {
                        String update1 = "ALTER TABLE sch0.table" + (j + 1) + " ADD COLUMN table" + j + "_id INT";
                        statement.executeUpdate(update1);
                        String update2;
                        if (dbType == DatabaseType.MYSQL || dbType == DatabaseType.POSTGRES) {
                            update2 = "ALTER TABLE sch0.table" + (j + 1) + " ADD CONSTRAINT id_constraint_" + (j + 1) + " FOREIGN KEY (table" + j + "_id) REFERENCES sch0.table" + j + "(id)";
                        } else {
                            throw new UnsupportedOperationException("Database type " + dbType + " not supported");
                        }

                        statement.executeUpdate(update2);
                    }

                    // Foreign key with multiple columns
                    statement.executeUpdate("CREATE TABLE sch0.table_x (idx1 INT, idx2 INT)");
                    statement.executeUpdate("ALTER TABLE sch0.table_x ADD CONSTRAINT pkx PRIMARY KEY (idx1, idx2)");
                    statement.executeUpdate("CREATE TABLE sch0.table_y (idy1 INT, idy2 INT)");
                    statement.executeUpdate("ALTER TABLE sch0.table_y ADD CONSTRAINT pky PRIMARY KEY (idy1, idy2)");
                    statement.executeUpdate("ALTER TABLE sch0.table_y ADD CONSTRAINT fky FOREIGN KEY(idy1, idy2) REFERENCES sch0.table_x(idx1, idx2)");

                    // Foreign keys between different schemas
                    statement.executeUpdate("ALTER TABLE sch0.table0 ADD COLUMN sch1table0id INT");
                    statement.executeUpdate("ALTER TABLE sch0.table0 ADD CONSTRAINT sch1table0id_constraint FOREIGN KEY(sch1table0id) REFERENCES sch1.table0(id)");
                    statement.executeUpdate("ALTER TABLE sch1.table0 ADD COLUMN sch0table0id INT");
                    statement.executeUpdate("ALTER TABLE sch1.table0 ADD CONSTRAINT sch0table0id_constraint FOREIGN KEY(sch0table0id) REFERENCES sch0.table0(id)");
                }
            }
        }
    }

    private boolean checkKey(Key key, String fromTableCompleteName, String[] fromColumns, String targetTableCompleteName, String[] targetColumns, boolean isVisible) {
        if (!fromTableCompleteName.equals(key.getFromTableCompleteName())) {
            return false;
        }
        if (!targetTableCompleteName.equals(key.getTargetTableCompleteName())) {
            return false;
        }
        for (int i = 0; i < key.getKeyColumns().size(); i++) {
            KeyColumn keyColumn = key.getKeyColumns().get(i);
            if (!fromColumns[i].equals(keyColumn.getFromColumn())) {
                return false;
            }
            if (!targetColumns[i].equals(keyColumn.getTargetColumn())) {
                return false;
            }
        }
        if (isVisible != key.isVisible()) {
            return false;
        }
        for (KeyColumn keyColumn : key.getKeyColumns()) {
            if (!Objects.equals(key.getId(), keyColumn.getKeyId())) {
                return false;
            }
        }
        return true;
    }

    private boolean checkKey(Key key, String fromTableCompleteName, String fromColumn, String targetTableCompleteName, String targetColumn, boolean isVisible) {
        return checkKey(key, fromTableCompleteName, new String[]{fromColumn}, targetTableCompleteName, new String[]{targetColumn}, isVisible);
    }

    @Test
    public void createNewAndUpdate() throws SQLException {
        LOG.info("TEST createNewAndUpdate STARTED");

        try {
            removeTestingDatabases();

            setUpTestingDatabases();

            for (DBWrapper dbWrapper : dbWrappers) {

                // Initializing a not existing TAP_SCHEMA
                TapSchema tapSchema = new TapSchema(dbWrapper, settings, false);

                /////////////////////////////////////
                //         ADDING A SCHEMA         //
                /////////////////////////////////////
                //
                Schema sch0 = tapSchema.addChild("sch0");
                assertEquals(Status.ADDED_NOT_PERSISTED, sch0.getStatus());

                Set<Key> allKeys = tapSchema.getAllKeys();
                LOG.debug("ALL keys:");
                for (Key key : allKeys) {
                    LOG.debug(key.toString());
                }

                // In the testing schemas each numbered table references the id 
                // of the previous table, except for the first table, so there
                // should be "TABLES_COUNT - 1" keys for the numbered tables.
                // Moreover, there are also the following keys:
                // - sch0.table0.sch1table0id -> sch1.table0.id
                // - sch1.table0.sch0table0id -> sch0.table0.id
                // - sch0.table_y.(idy1, idy2) -> sch0.table_x.(idyx, idx2)
                // so we check for TABLES_COUNT + 2.
                assertEquals(TABLES_COUNT + 2, allKeys.size());

                // Checking that keys information has been filled correctly.
                for (Key schemaKey : allKeys) {
                    assertFalse(schemaKey.isVisible());
                    assertNull(schemaKey.getId());
                    assertEquals(schemaKey.getFromTableCompleteName(), schemaKey.getFromSchemaName() + "." + schemaKey.getFromTableSimpleName());
                    assertEquals(schemaKey.getTargetTableCompleteName(), schemaKey.getTargetSchemaName() + "." + schemaKey.getTargetTableSimpleName());
                    assertTrue(schemaKey.getKeyColumns().size() >= 1);
                    for (KeyColumn keyColumn : schemaKey.getKeyColumns()) {
                        assertNotNull(keyColumn.getFromColumn());
                        assertNotNull(keyColumn.getTargetColumn());
                    }
                }
                /////////////////////////////////////
                //         ADDING A TABLE          //
                /////////////////////////////////////
                //
                Table sch0table0 = sch0.addChild("table0");
                assertEquals(Status.ADDED_NOT_PERSISTED, sch0table0.getStatus());
                assertEquals("sch0.table0", sch0table0.getCompleteName());
                assertEquals(1, sch0.getChildren().size());

                assertEquals(4, sch0table0.getAddableChildrenNames().size());

                Column sch0table0id = sch0table0.addChild("id");
                assertEquals(Status.ADDED_NOT_PERSISTED, sch0table0id.getStatus());

                // Primary key check
                assertTrue(sch0table0id.isPrimaryKey());

                /////////////////////////////////////
                //         KEYS MANAGEMENT         //
                /////////////////////////////////////
                //
                // CASE 1: Foreign key between two tables in the same schema.
                // sch0.table1.t0id -> sch0.table0.id
                //
                // Adding sch0.table1
                Table sch0table1 = sch0.addChild("table1");
                assertEquals(2, sch0.getChildren().size());

                assertTrue(tapSchema.getVisibileKeys().isEmpty());
                sch0table1.addChild("table0_id");
                assertEquals(1, tapSchema.getVisibileKeys().size());

                // Check if key and its columns have been properly initialized
                Key k1 = tapSchema.getVisibileKeys().get(0);
                assertTrue(checkKey(k1, "sch0.table1", "table0_id", "sch0.table0", "id", true));

                // Removing sch0.table1
                sch0.removeChild("table1");
                assertEquals(1, sch0.getAddedChildren().size());
                assertTrue(tapSchema.getVisibileKeys().isEmpty());

                //
                // CASE 2: Foreign key between two tables in different schemas.
                // sch1.table0.sch0table0id -> sch0.table0.id
                //
                // Adding sch1
                Schema sch1 = tapSchema.addChild("sch1");
                assertEquals(5, allKeys.size());

                // Adding sch1.table0
                Table sch1table0 = sch1.addChild("table0");

                sch1table0.addChild("id");
                sch1table0.addChild("sch0table0id"); // sch1.table0.sch0table0id -> sch0.table0.it obtains keyId = "1"
                sch0table0.addChild("sch1table0id");// sch0.table0.sch1table0id -> sch1.table0.it obtains keyId = "2"

                assertEquals(2, tapSchema.getVisibileKeys().size());

                for (Key key : tapSchema.getVisibileKeys()) {
                    assertTrue(
                            checkKey(key, "sch0.table1", "table0_id", "sch0.table0", "id", false)
                            || checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true)
                            || checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true)
                    );
                }

                // Removing sch1
                tapSchema.removeChild("sch1");
                assertEquals(Status.REMOVED_NOT_PERSISTED, sch1.getStatus());
                assertEquals(0, tapSchema.getVisibileKeys().size());

                // Case 2B: Re-adding sch1
                // sch1.table0 has not been removed from its schema, so the keys
                // should be re-added.
                tapSchema.addChild("sch1");
                assertEquals(Status.ADDED_NOT_PERSISTED, sch1.getStatus());
                assertEquals(2, tapSchema.getVisibileKeys().size());

                //
                // CASE 3: foreign key with multiple columns
                //
                Table table_x = sch0.addChild("table_x");
                Table table_y = sch0.addChild("table_y");
                assertEquals(2, tapSchema.getVisibileKeys().size());

                table_x.addChild("idx1");
                table_x.addChild("idx2");
                table_y.addChild("idy1");
                assertEquals(2, tapSchema.getVisibileKeys().size());

                table_y.addChild("idy2");
                assertEquals(3, tapSchema.getVisibileKeys().size());

                for (Key key : tapSchema.getVisibileKeys()) {
                    assertTrue(
                            checkKey(key, "sch0.table1", "table0_id", "sch0.table0", "id", false)
                            || checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true)
                            || checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true)
                            || checkKey(key, "sch0.table_y", new String[]{"idy1", "idy2"}, "sch0.table_x", new String[]{"idx1", "idx2"}, true)
                    );
                }

                /////////////////////////////////////
                //              SAVE               //
                /////////////////////////////////////
                sch0table1 = sch0.addChild("table1");
                sch0table1.addChild("id");
                Table sch0table2 = sch0.addChild("table2");
                sch0table2.addChild("table1_id");

                UpdateOperations operations = new UpdateOperations(tapSchema);
                assertEquals(2, operations.getSchemasToAdd().size());
                assertEquals(6, operations.getTablesToAdd().size());
                assertEquals(11, operations.getColumnsToAdd().size());
                assertEquals(5, operations.getKeysToAdd().size());

                tapSchema.save();
                operations = new UpdateOperations(tapSchema);
                assertFalse(operations.getHasOperations());

                // Checking size and arraysize detection
                Schema tapSchemaSchema = tapSchema.getChild(TapSchema.STANDARD_TAP_SCHEMA_NAME);
                assertEquals(tapSchema.getDBName(), tapSchemaSchema.getValue(Schema.DBNAME));
                assertEquals(TapSchema.STANDARD_TAP_SCHEMA_NAME, tapSchemaSchema.getName());

                Table columnsTable = tapSchemaSchema.getChild("columns");

                Column utypeColumn = columnsTable.getChild("utype");
                assertEquals("char", utypeColumn.getValue("datatype", String.class));
                Integer size = utypeColumn.getValue("size", Integer.class);
                String arraySize = utypeColumn.getValue("arraysize", String.class);
                assertTrue(size == null || size == 255); // null -> Postgres, 255 -> MySQL
                assertTrue("*".equals(arraySize) || "255*".equals(arraySize)); // * -> Postgres, 255* -> MySQL

                // reloading
                LOG.debug("----- Reloading saved TAP_SCHEMA -----");
                tapSchema = new TapSchema(dbWrapper, settings, true);
                assertEquals("1.1-IA2", tapSchema.getTapSchemaDBBroker().detectVersion(tapSchema.getRealName()));
                LOG.debug(tapSchema.toString());

                assertNotNull(sch0 = tapSchema.getChild("sch0", Status.ADDED_PERSISTED));
                assertNotNull(sch1 = tapSchema.getChild("sch1", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0 = sch0.getChild("table0", Status.ADDED_PERSISTED));
                assertNotNull(sch0table1 = sch0.getChild("table1", Status.ADDED_PERSISTED));
                assertNotNull(sch0table2 = sch0.getChild("table2", Status.ADDED_PERSISTED));
                assertNotNull(table_x = sch0.getChild("table_x", Status.ADDED_PERSISTED));
                assertNotNull(table_y = sch0.getChild("table_y", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0 = sch1.getChild("table0", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0.getChild("sch1table0id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table1.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table1.getChild("table0_id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table2.getChild("table1_id", Status.ADDED_PERSISTED));
                assertNotNull(table_x.getChild("idx1", Status.ADDED_PERSISTED));
                assertNotNull(table_x.getChild("idx2", Status.ADDED_PERSISTED));
                assertNotNull(table_y.getChild("idy1", Status.ADDED_PERSISTED));
                assertNotNull(table_y.getChild("idy2", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0.getChild("sch0table0id", Status.ADDED_PERSISTED));

//                for (Key key : allKeys) {
//                    if (key.getId() == null) {
//                        assertFalse(key.isVisible());
//                    } else {
//                        assertTrue(
//                                checkKey(key, "sch0.table1", "table0_id", "sch0.table0", "id", true)
//                                || checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true)
//                                || checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true)
//                                || checkKey(key, "sch0.table_y", new String[]{"idy1", "idy2"}, "sch0.table_x", new String[]{"idx1", "idx2"}, true)
//                                || checkKey(key, "sch0.table2", "table1_id", "sch0.table1", "id", true)
//                        );
//                    }
//                }
                sch0.removeChild("table1");

                operations = new UpdateOperations(tapSchema);
                assertFalse(operations.getHasEntitiesToUpdate());
                assertFalse(operations.getHasEntitiesToAdd());
                assertEquals(0, operations.getSchemasToRemove().size());
                assertEquals(1, operations.getTablesToRemove().size());
                assertEquals(2, operations.getColumnsToRemove().size());
                assertEquals(2, operations.getKeysToRemove().size());

                tapSchema.save();
                LOG.debug(tapSchema.toString());
                operations = new UpdateOperations(tapSchema);
                assertFalse(operations.getHasOperations());

                // reloading
                LOG.debug("----- Reloading saved TAP_SCHEMA -----");
                tapSchema = new TapSchema(dbWrapper, settings, true);
                LOG.debug(tapSchema.toString());

                assertNotNull(sch0 = tapSchema.getChild("sch0", Status.ADDED_PERSISTED));
                assertNotNull(sch1 = tapSchema.getChild("sch1", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0 = sch0.getChild("table0", Status.ADDED_PERSISTED));
                assertNotNull(sch0table2 = sch0.getChild("table2", Status.ADDED_PERSISTED));
                assertNotNull(table_x = sch0.getChild("table_x", Status.ADDED_PERSISTED));
                assertNotNull(table_y = sch0.getChild("table_y", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0 = sch1.getChild("table0", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table0.getChild("sch1table0id", Status.ADDED_PERSISTED));
                assertNotNull(sch0table2.getChild("table1_id", Status.ADDED_PERSISTED));
                assertNotNull(table_x.getChild("idx1", Status.ADDED_PERSISTED));
                assertNotNull(table_x.getChild("idx2", Status.ADDED_PERSISTED));
                assertNotNull(table_y.getChild("idy1", Status.ADDED_PERSISTED));
                assertNotNull(table_y.getChild("idy2", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0.getChild("id", Status.ADDED_PERSISTED));
                assertNotNull(sch1table0.getChild("sch0table0id", Status.ADDED_PERSISTED));

//                for (Key key : allKeys) {
//                    if (key.getId() == null) {
//                        assertFalse(key.isVisible());
//                    } else {
//                        assertNotNull(key.getOriginalValue(Key.ID_KEY, String.class)); // for reloaded keys
//                        assertTrue(
//                                checkKey(key, "sch1.table0", "sch0table0id", "sch0.table0", "id", true)
//                                || checkKey(key, "sch0.table0", "sch1table0id", "sch1.table0", "id", true)
//                                || checkKey(key, "sch0.table_y", new String[]{"idy1", "idy2"}, "sch0.table_x", new String[]{"idx1", "idx2"}, true)
//                                || checkKey(key, "sch0.table2", "table1_id", "sch0.table1", "id", true)
//                        );
//                    }
//                }
                // Test adding ficitious key
//                sch0table0.addChild("value1");
//                sch0table2.addChild("value1");
//                ((TapSchema) tapSchema).addFictitiousKey(sch0table2, new String[]{"value1"}, sch0table0, new String[]{"value1"});
//                operations = new UpdateOperations(tapSchema);
//                assertEquals(1, operations.getKeysToAdd().size());
//                tapSchema.save();
//
//                tapSchema = TapSchemaFactory.getTapSchema(TapSchemaVersion.TAP_SCHEMA_1_IA2, dbWrapper, "test_tap_schema", true);
//
//                sch0table2 = tapSchema.getChild("sch0").getChild("table2");
//                assertEquals(1, sch0table2.getVisibleFromKeys().size());
//                checkKey(sch0table2.getVisibleFromKeys().get(0), "sch0.table2", "value1", "sch0.table0", "value1", true);
//
//                tapSchema.removeChild("sch1");
//                assertEquals(1, tapSchema.getChildren(Status.TO_REMOVE).size());
//                tapSchema.save();
//                assertEquals(0, tapSchema.getChildren(Status.TO_REMOVE).size());
                /////////////////////////////////////
                //       CONSISTENCY CHECKS        //
                /////////////////////////////////////
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            //removeTestingDatabases(credentials);
        }
    }

    @Test
    public void testTapSchemaSerialization() throws Exception {
        for (DBWrapper dbWrapper : dbWrappers) {
            if (dbWrapper.getTapSchemaDatabaseType() == DatabaseType.MYSQL) { // currently "tng_TAP_SCHEMA" exists in a MySQL instance
                TapSchema tapSchema = new TapSchema(dbWrapper, settings, true);

                File temp = File.createTempFile("test_tap_schema", ".ser");

                try (FileOutputStream fileOut = new FileOutputStream(temp);
                        ObjectOutputStream out = new ObjectOutputStream(fileOut)) {
                    out.writeObject(tapSchema);
                }

                try (FileInputStream fileIn = new FileInputStream(temp);
                        ObjectInputStream in = new ObjectInputStream(fileIn)) {
                    tapSchema = (TapSchema) in.readObject();
                }

                LOG.debug(tapSchema.toString());
            }
        }
    }
}
