/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.datalayer;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.SQLException;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is used to silently manage the possibility to have separate data
 * sources for the TAP_SCHEMA schema and its <em>source</em> schema (the schema
 * from which it takes the information, that is the schema containing the
 * astronomical data). Also the {@code ivoa} schema, containing the
 * {@code ObsCore} table, is stored into the source database (because one could
 * perform SQL JOINS between astronomical data schemata and the ObsCore table).
 * <p>
 * An user asks, for example, {@link #getSourceDataSource()}, and the
 * {@code DBWrapper} returns the correct {@code DataSource}, both if it is
 * separated from the TAP_SCHEMA {@code DataSource} or if they are the same
 * object.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class DBWrapper implements Serializable {

    private static final long serialVersionUID = 1721030677924066695L;
    private final static Logger LOG = LoggerFactory.getLogger(DBWrapper.class);

    // Same credentials (this is null if the credentials are separated)
    private DataSourceWrapper dataSourceWrapper;

    // Separated credentials (these are null if the credentials are the same)
    private DataSourceWrapper sourceDataSourceWrapper;
    private DataSourceWrapper tapSchemaDataSourceWrapper;

    private DBWrapper() {
    }

    /**
     * Constructor to use if the source schema credentials and the TAP_SCHEMA
     * credentials are the same.
     */
    public DBWrapper(Credentials credentials) {
        this();
        dataSourceWrapper = new DataSourceWrapper(credentials);
    }

    /**
     * Constructor to use if the source schema credentials are different from
     * the TAP_SCHEMA credentials.
     */
    public DBWrapper(Credentials sourceCredentials, Credentials tapSchemaCredentials) {
        this();
        sourceDataSourceWrapper = new DataSourceWrapper(sourceCredentials);
        tapSchemaDataSourceWrapper = new DataSourceWrapper(tapSchemaCredentials);
    }

    public DataSourceWrapper getSourceDataSourceWrapper() {
        if (dataSourceWrapper != null) {
            return dataSourceWrapper;
        }
        return sourceDataSourceWrapper;
    }

    public DataSource getSourceDataSource() {
        return getSourceDataSourceWrapper().getDataSource();
    }

    public DataSourceWrapper getTapSchemaDataSourceWrapper() {
        if (dataSourceWrapper != null) {
            return dataSourceWrapper;
        }
        return tapSchemaDataSourceWrapper;
    }

    public DataSource getTapSchemaDataSource() {
        return getTapSchemaDataSourceWrapper().getDataSource();
    }

    public Connection getSourceConnection() throws SQLException {
        return getSourceDataSource().getConnection();
    }

    public Connection getTapSchemaConnection() throws SQLException {
        return getTapSchemaDataSource().getConnection();
    }

    public Credentials getSourceCredentials() {
        return getSourceDataSourceWrapper().getCredentials();
    }

    public Credentials getTapSchemaCredentials() {
        return getTapSchemaDataSourceWrapper().getCredentials();
    }

    public DatabaseType getSourceDatabaseType() {
        return getSourceDataSourceWrapper().getDatabaseType();
    }

    public DatabaseType getTapSchemaDatabaseType() {
        return getTapSchemaDataSourceWrapper().getDatabaseType();
    }

    /**
     * Tells if the {@code DataSource}s for the TAP_SCHEMA and its source are
     * different.
     *
     * @return true if the TAP_SCHEMA {@code DataSource} is different from its
     * source {@code DataSource}, false otherwise.
     */
    public boolean isSeparatedSources() {
        return dataSourceWrapper == null;
    }

    /**
     * Test both the connection to the TAP_SCHEMA {@code DataSource} and its
     * source {@code DataSource}.
     *
     * @throws SQLException if it is not possible to connect to the
     * {@code DataSource}.
     */
    public void testConnections() throws SQLException {
        Connection connection;

        if (isSeparatedSources()) {
            connection = getSourceDataSource().getConnection();
            connection.close();
            connection = getTapSchemaDataSource().getConnection();
            connection.close();
        } else {
            connection = getSourceDataSource().getConnection();
            connection.close();
        }
    }
}
