/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.model;

import it.inaf.ia2.tsm.datalayer.ADQL;
import it.inaf.ia2.tsm.datalayer.DataTypeMode;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import javax.xml.bind.JAXB;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElements;
import javax.xml.bind.annotation.XmlRootElement;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class TypesMapping {

    @XmlRootElement(name = "sql_type_mapping")
    static class TypesWrapper {

        private List<TypeMapping> types;

        @XmlElements({
            @XmlElement(name = "type")
        })
        public List<TypeMapping> getTypes() {
            return types;
        }

        public void setTypes(List<TypeMapping> types) {
            this.types = types;
        }
    }

    private static final List<TypeMapping> TYPES;

    static {
        try (InputStream is = TypeMapping.class.getClassLoader().getResourceAsStream("sql_type_mapping.xml")) {

            TypesWrapper typesWrapper = JAXB.unmarshal(is, TypesWrapper.class);
            TYPES = typesWrapper.types;
        } catch (IOException e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    private static TypeMapping getTypeMappingFromADQLType(String adqlType) {
        for (TypeMapping typeMapping : TYPES) {
            if (adqlType.equals(typeMapping.getAdqlType())) {
                return typeMapping;
            }
        }
        return null;
    }

    public static Class getClassFromAdqlType(String adqlType) {
        TypeMapping typeMapping = getTypeMappingFromADQLType(adqlType);
        if (typeMapping == null) {
            return String.class;
        }
        return typeMapping.getJavaType();
    }

    public static String getMySQLTypeFromADQLType(String adqlType) {
        TypeMapping typeMapping = getTypeMappingFromADQLType(adqlType);
        if (typeMapping == null) {
            return "VARCHAR";
        }
        return typeMapping.getMySQLMapping().getTypes().get(0);
    }

    public static String getPostgresSQLTypeFromADQLType(String adqlType) {
        TypeMapping typeMapping = getTypeMappingFromADQLType(adqlType);
        if (typeMapping == null) {
            return "character varying";
        }
        return typeMapping.getPgsqlMapping().getTypes().get(0);
    }

    public static String getDataType(String adqlType, DataTypeMode mode) {
        switch (mode) {
            case ADQL:
                return adqlType;
            case VOTABLE:
                return getTypeMappingFromADQLType(adqlType).getVoTableType();
            default:
                throw new UnsupportedOperationException("DataTypeMode " + mode + " not supported yet");
        }
    }

    public static TypeMapping getTypeMapping(String dataType, DataTypeMode mode) {
        for (TypeMapping typeMapping : TYPES) {
            switch (mode) {
                case ADQL:
                    if (dataType.equals(typeMapping.getAdqlType())) {
                        return typeMapping;
                    }
                    break;
                case VOTABLE:
                    if (dataType.equals(typeMapping.getVoTableType())) {
                        return typeMapping;
                    }
                    break;
                default:
                    throw new UnsupportedOperationException("DataTypeMode " + mode + " not supported yet");
            }
        }
        throw new RuntimeException("Unable to get TypeMapping for datatype " + dataType + " [" + mode + "]");
    }

    private static abstract class DataTypeFromDBTypeMapper {

        private final DataTypeMode mode;

        DataTypeFromDBTypeMapper(DataTypeMode mode) {
            this.mode = mode;
        }

        abstract DBTypeMapping getDBTypeMapping(TypeMapping typeMapping);

        private String getDataType(TypeMapping mapping) {
            switch (mode) {
                case ADQL:
                    return mapping.getAdqlType();
                case VOTABLE:
                    return mapping.getVoTableType();
                default:
                    throw new UnsupportedOperationException("DataTypeMode " + mode + " not supported yet");
            }
        }

        private String getDefaultDataType(String dbType) {
            switch (mode) {
                case ADQL:
                    return ADQL.getDataType(dbType);
                case VOTABLE:
                    return "char";
                default:
                    throw new UnsupportedOperationException("DataTypeMode " + mode + " not supported yet");
            }
        }

        String getDataTypeFromDBType(String dbType) {
            String dbTypeWithoutSize = dbType.toUpperCase().replaceAll("\\(.+\\)", "");
            for (TypeMapping typeMapping : TYPES) {
                DBTypeMapping dbMapping = getDBTypeMapping(typeMapping);
                if (dbMapping.isInverse()) {
                    for (String type : dbMapping.getTypes()) {
                        if (dbTypeWithoutSize.toLowerCase().startsWith(type.toLowerCase())) {
                            return getDataType(typeMapping);
                        }
                    }
                }
            }
            return getDefaultDataType(dbTypeWithoutSize);
        }
    }

    public static String getDataTypeFromPostgresType(String pgsqlType, DataTypeMode mode) {

        return (new DataTypeFromDBTypeMapper(mode) {
            @Override
            DBTypeMapping getDBTypeMapping(TypeMapping typeMapping) {
                return typeMapping.getPgsqlMapping();
            }
        }).getDataTypeFromDBType(pgsqlType);
    }

    public static ADQL getADQLFromDBType(String dbType) {
        try {
            return ADQL.valueOf(TypesMapping.getDataTypeFromPostgresType(dbType, DataTypeMode.ADQL));
        } catch (IllegalArgumentException e) {
            return null;
        }
    }

    public static String getDataTypeFromMySQLType(String mysqlType, DataTypeMode mode) {

        return (new DataTypeFromDBTypeMapper(mode) {
            @Override
            DBTypeMapping getDBTypeMapping(TypeMapping typeMapping) {
                return typeMapping.getMySQLMapping();
            }
        }).getDataTypeFromDBType(mysqlType);
    }

    public static Object parseDefaultValue(String defaultValue, String type) {
        return parseDefaultValue(defaultValue, ADQL.valueOf(type));
    }

    public static Object parseDefaultValue(String defaultValue, ADQL type) {

        switch (type) {
            case BOOLEAN:
                return Boolean.parseBoolean(defaultValue);
            case SMALLINT:
                return Short.parseShort(defaultValue);
            case INTEGER:
                return Integer.parseInt(defaultValue);
            case BIGINT:
                return Long.parseLong(defaultValue);
            case REAL:
                return Float.parseFloat(defaultValue);
            case DOUBLE:
                return Double.parseDouble(defaultValue);
            default:
                return defaultValue;
        }
    }
}
