/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import it.inaf.ia2.tsm.model.ColumnModel;
import it.inaf.ia2.tsm.model.TableModel;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Represents an object that is mapped on a table of the TAP_SCHEMA.
 *
 * A TapSchemaEntity has some properties that correspond to columns of the table
 * represented by the TapSchemaEntity.<br>
 * Property value can be changed but the original value has to be maintained
 * until the {@link #save()} method is called.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public abstract class TapSchemaEntity implements Serializable {

    private static final long serialVersionUID = 5515596028279668709L;
    private static final Logger LOG = LoggerFactory.getLogger(TapSchemaEntity.class);

    private Map<String, Object> metadata;
    private Map<String, EntityProperty> properties;
    protected TapSchema tapSchema;
    private TableModel tableModel;

    protected TapSchemaEntity() {
    }

    public TapSchemaEntity(TapSchema tapSchema, TableModel tableModel, Map<String, Object> metadata) {
        this.tapSchema = tapSchema;
        this.tableModel = tableModel;
        this.metadata = metadata;
        this.properties = new HashMap<>();
        fillProperties();
    }

    private void fillProperties() {
        for (ColumnModel propModel : tableModel.getColumns()) {
            Object defaultValue = null;
            if (propModel.getLoaderKey() != null) {
                defaultValue = metadata.get(propModel.getLoaderKey());

                if (defaultValue != null) {
                    // Special case for boolean to integer conversion
                    Class javaType = propModel.getJavaType();
                    if (javaType != defaultValue.getClass()) {
                        if (defaultValue.getClass() == Boolean.class && javaType == Integer.class) {
                            defaultValue = ((Boolean) defaultValue) ? 1 : 0;
                        } else {
                            throw new UnsupportedOperationException("Unable to convert " + defaultValue.getClass().getCanonicalName() + " into " + propModel.getType());
                        }
                    }
                }
            }
            EntityProperty ep = new EntityProperty(propModel, defaultValue);
            this.properties.put(propModel.getName(), ep);
        }
    }

    public Object getMetadata(String key) {
        return metadata.get(key);
    }

    /**
     * Initializes the value of a property (store the original value).
     */
    public <T> void initProperty(String key, T value) {
        properties.get(key).init(value);
    }

    protected String getVersion() {
        return tapSchema.getVersion();
    }

    /**
     * Returns true if one or more property values is changed (the current value
     * is different from the original value).
     */
    public boolean isChanged() {
        for (EntityProperty property : properties.values()) {
            if (property.isChanged()) {
                return true;
            }
        }
        return false;
    }

    public Object getValue(String key) {
        return properties.get(key).getValue();
    }

    public Class getPropertyType(String key) {
        return properties.get(key).getType();
    }

    public EntityProperty getProperty(String key) {
        return properties.get(key);
    }

    /**
     * Retrieve the current value of the property (the last value set).
     */
    public <T> T getValue(String key, Class<T> type) {
        return (T) properties.get(key).getValue();
    }

    public void setValue(String key, Object value) {
        properties.get(key).setValue(value);
    }

    /**
     * Retrieve the original value of the property.
     *
     * @param key the name of the property (the name of the table column).
     * @param type the class of the property value.
     * @return
     */
    public <T> T getOriginalValue(String key, Class<T> type) {
        return (T) properties.get(key).getOriginalValue();
    }

    /**
     * Returns true the value of property which key is passed as parameter is
     * changed (the current value is different from the original value).
     */
    public boolean isChanged(String key) {
        return properties.get(key).isChanged();
    }

    /**
     * Retrieve a list of all properties names (the names of the table columns).
     */
    public List<String> getPropertiesKeys() {
        return new ArrayList<>(properties.keySet());
    }

    /**
     * Marks the TapSchemaEntity as saved (all original values are set equals to
     * current values).
     */
    public void save() {
        for (EntityProperty p : properties.values()) {
            p.save();
        }
    }

    /**
     * Set the correct value for a fixed property that has inconsistent value in
     * the TAP_SCHEMA.
     */
    public <T> void amendProperty(String key, T value) {
        EntityProperty prop = properties.get(key);
        prop.init(value);
        prop.setChanged(true);
    }

    public TableModel getTableModel() {
        return tableModel;
    }
}
