/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.model;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.xml.bind.JAXB;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class SchemaModels {

    private static final String SCHEMA_DEFINITION_FOLDER = "schema_definition";
    private static final String TAP_SCHEMA_NAME = "tap_schema";
    private static final String IVOA_SCHEMA_NAME = "ivoa";

    // Map keys are version numbers
    private static final Map<String, SchemaModel> TAP_SCHEMA_MODELS;
    private static final Map<String, SchemaModel> IVOA_SCHEMA_MODELS;

    private SchemaModels() {
    }

    static {
        try {
            String[] xmlModelFileNames = getXMLModelFileNames();
            List<SchemaXMLModel> xmlModels = getXmlModels(xmlModelFileNames);

            Map<String, SchemaXMLModel> tapSchemaXmlModelsMap = getXmlModelMapByVersionAndName(xmlModels, TAP_SCHEMA_NAME);
            Map<String, SchemaXMLModel> ivoaSchemaXmlModelsMap = getXmlModelMapByVersionAndName(xmlModels, IVOA_SCHEMA_NAME);

            TAP_SCHEMA_MODELS = getModelMapByVersion(tapSchemaXmlModelsMap);
            IVOA_SCHEMA_MODELS = getModelMapByVersion(ivoaSchemaXmlModelsMap);
        } catch (IOException e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    private static String[] getXMLModelFileNames() throws IOException {
        try (InputStream in = SchemaModels.class.getClassLoader().getResourceAsStream("core.properties")) {

            Properties props = new Properties();
            props.load(in);

            String[] fileNames = props.getProperty("schema_definition_files").split(",");

            String[] modelFiles = new String[fileNames.length];
            for (int i = 0; i < fileNames.length; i++) {
                modelFiles[i] = String.format("%s%s%s", SCHEMA_DEFINITION_FOLDER, File.separator, fileNames[i]);
            }

            return modelFiles;
        }
    }

    private static List<SchemaXMLModel> getXmlModels(String[] xmlModelFileNames) throws IOException {
        List<SchemaXMLModel> xmlModels = new ArrayList<>();

        for (String modelFile : xmlModelFileNames) {
            try (InputStream in = SchemaModels.class.getClassLoader().getResourceAsStream(modelFile)) {
                SchemaXMLModel model = JAXB.unmarshal(in, SchemaXMLModel.class);
                xmlModels.add(model);
            }
        }

        return xmlModels;
    }

    private static void loadSchemaModel(SchemaModel model, SchemaXMLModel xmlModel, Map<String, SchemaXMLModel> xmlModels) {

        for (TableXMLModel tableXmlModel : xmlModel.getTables()) {
            String tableName = tableXmlModel.getName();
            TableModel tableModel = model.get(tableName);
            if (tableModel == null) {
                tableModel = new TableModel(tableXmlModel);
            }
            for (PropertyModel property : tableXmlModel.getAdd()) {
                tableModel.getProperties().put(property.getName(), property);
            }
            model.getTables().put(tableName, tableModel);
        }

        if (xmlModel.getExtendsFrom() != null) {
            SchemaXMLModel parentModel = xmlModels.get(xmlModel.getExtendsFrom());
            loadSchemaModel(model, parentModel, xmlModels);
        }
    }

    private static Map<String, SchemaXMLModel> getXmlModelMapByVersionAndName(List<SchemaXMLModel> xmlModels, String schemaName) {
        Map<String, SchemaXMLModel> map = new HashMap<>();

        for (SchemaXMLModel xmlModel : xmlModels) {
            if (schemaName.equals(xmlModel.getName())) {
                map.put(xmlModel.getVersion(), xmlModel);
            }
        }

        return map;
    }

    private static Map<String, SchemaModel> getModelMapByVersion(Map<String, SchemaXMLModel> xmlModels) {
        Map<String, SchemaModel> map = new HashMap<>();
        for (SchemaXMLModel xmlModel : xmlModels.values()) {
            SchemaModel model = new SchemaModel(xmlModel);
            loadSchemaModel(model, xmlModel, xmlModels);
            map.put(model.getVersion(), model);
        }
        return map;
    }

    public static SchemaModel getTapSchemaModel(String version) {
        return TAP_SCHEMA_MODELS.get(version);
    }

    public static Iterable<SchemaModel> getTapSchemaModels() {
        return TAP_SCHEMA_MODELS.values();
    }

    public static TableModel getTapSchemaTableModel(String tableName, String version) {
        return getTapSchemaModel(version).getTables().get(tableName);
    }

    private static List<String> getAvailableVersions(Map<String, SchemaModel> schemaModels) {
        List<String> versions = new ArrayList<>();
        for (SchemaModel tapSchemaModel : schemaModels.values()) {
            versions.add(tapSchemaModel.getVersion());
        }
        return versions;
    }

    public static List<String> getAvailableTapSchemaVersions() {
        return getAvailableVersions(TAP_SCHEMA_MODELS);
    }

    public static List<String> getAvailableIvoaSchemaVersions() {
        return getAvailableVersions(IVOA_SCHEMA_MODELS);
    }

    public static SchemaModel getIvoaSchemaModel(String version) {
        return IVOA_SCHEMA_MODELS.get(version);
    }

    public static Iterable<SchemaModel> getIvoaSchemaModels() {
        return IVOA_SCHEMA_MODELS.values();
    }
}
