/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import it.inaf.ia2.tsm.TapSchema;
import it.inaf.ia2.tsm.webapp.xmlconfig.SeparatedCredentials;
import it.inaf.ia2.tsm.datalayer.Credentials;
import it.inaf.ia2.tsm.model.SchemaModels;
import it.inaf.ia2.tsm.webapp.xmlconfig.JoinedCredentials;
import it.inaf.ia2.tsm.webapp.xmlconfig.TapCredentials;
import java.io.IOException;
import java.io.Serializable;
import java.sql.SQLException;
import java.util.List;
import javax.annotation.PostConstruct;
import javax.faces.application.FacesMessage;
import javax.faces.context.FacesContext;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.deltaspike.core.api.scope.WindowScoped;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named("credentialsInsertion")
@WindowScoped
public class CredentialsEditing implements Serializable {

    private static final long serialVersionUID = -2688980249773483198L;
    private static final Logger LOG = LoggerFactory.getLogger(CredentialsEditing.class);

    @Inject
    private User user;

    @Inject
    private ConfigurationManager config;

    @Inject
    private TapSchemaLoader tapSchemaLoader;

    private Integer credentialsInEditing;
    private Integer credentialsToRemove;

    private boolean credentialsDialogOpened;

    private List<String> tapSchemaVersions;
    private List<String> obscoreVersions;

    private String label;
    private boolean separateCredentials;
    private Credentials sourceCredentials;
    private Credentials tapSchemaCredentials;
    private String tapSchemaName;
    private String tapSchemaVersion;
    private boolean hasObscore;
    private String obscoreVersion;
    private String ivoaSchemaName;

    @PostConstruct
    public void init() {
        this.tapSchemaVersions = SchemaModels.getAvailableTapSchemaVersions();
        obscoreVersions = SchemaModels.getAvailableIvoaSchemaVersions();
    }

    public List<TapCredentials> getSavedCredentials() {
        return user.getUserConfiguration().getCredentialsInfo();
    }

    public void editCredentials(int index) {
        credentialsInEditing = index;
        TapCredentials credentials = getSavedCredentials().get(index);

        label = credentials.getLabel();

        if (credentials instanceof JoinedCredentials) {

            JoinedCredentials joinedCredentials = (JoinedCredentials) credentials;
            separateCredentials = false;
            sourceCredentials = new Credentials(joinedCredentials.getCredentials());
            // in case the user wants to split credentials
            tapSchemaCredentials = new Credentials();

        } else if (credentials instanceof SeparatedCredentials) {

            SeparatedCredentials separatedCredentials = (SeparatedCredentials) credentials;
            separateCredentials = true;
            sourceCredentials = new Credentials(separatedCredentials.getSourceCredentials());
            tapSchemaCredentials = new Credentials(separatedCredentials.getTapSchemaCredentials());
        }

        tapSchemaName = credentials.getTapSchemaName();
        tapSchemaVersion = credentials.getTapSchemaVersion();
        hasObscore = credentials.isHasObscore();
        obscoreVersion = credentials.getObscoreVersion();
        ivoaSchemaName = credentials.getIvoaSchemaName();
    }

    public void addNewCredentialsInEditing() {
        credentialsInEditing = null;
        separateCredentials = false;
        sourceCredentials = new Credentials();
        tapSchemaCredentials = new Credentials();
        tapSchemaName = TapSchema.STANDARD_TAP_SCHEMA_NAME;
        tapSchemaVersion = "";
        hasObscore = false;
        obscoreVersion = "";
        ivoaSchemaName = TapSchema.STANDARD_IVOA_SCHEMA_NAME;
    }

    public void loadTapSchema(TapCredentials tapCredentials) throws SQLException {
        tapSchemaLoader.tryLoadingTapSchema(tapCredentials);
    }

    public void openDeleteCredentialsConfirmation(int index) throws IOException {
        credentialsToRemove = index;
    }

    public void confirmCredentialsDeletion() {
        if (credentialsToRemove != null) {
            config.deleteCredentials(user.getUsername(), credentialsToRemove);
        }
        credentialsToRemove = null;
    }

    public boolean validateNotNull(FacesContext ctx, String value, String componentId, String errorMessage) {
        if (value == null || value.isEmpty()) {
            ctx.addMessage(componentId, new FacesMessage(errorMessage));
            return false;
        }
        return true;
    }

    public void saveCredentialsEdited() throws IOException {

        /**
         * We need to validate manually to avoid problem with JSF AJAX partial
         * updates.
         */
        FacesContext ctx = FacesContext.getCurrentInstance();

        /**
         * Single & operator used to perform all validation.
         */
        boolean validationOk
                = validateNotNull(ctx, sourceCredentials.getHostname(), "main:source_hostname", "Hostname is required")
                & validateNotNull(ctx, sourceCredentials.getUsername(), "main:source_username", "Username is required")
                & validateNotNull(ctx, tapSchemaName, "main:tap_schema_name", "TAP_SCHEMA name is required");

        if (separateCredentials) {
            validationOk = validationOk
                    & validateNotNull(ctx, tapSchemaCredentials.getHostname(), "main:tap_schema_hostname", "Hostname is required")
                    & validateNotNull(ctx, tapSchemaCredentials.getHostname(), "main:tap_schema_username", "Username is required");
        }

        if (hasObscore) {
            validationOk = validationOk
                    & validateNotNull(ctx, ivoaSchemaName, "main:ivoa_schema_name", "IVOA schema name is required");
        }

        if (!validationOk) {
            return;
        }

        TapCredentials editedCredentials;
        if (separateCredentials) {
            editedCredentials = new SeparatedCredentials(sourceCredentials, tapSchemaCredentials);
        } else {
            editedCredentials = new JoinedCredentials(sourceCredentials);
        }

        editedCredentials.setLabel(label);
        editedCredentials.setTapSchemaName(tapSchemaName);
        editedCredentials.setTapSchemaVersion(tapSchemaVersion);
        editedCredentials.setHasObscore(hasObscore);
        editedCredentials.setObscoreVersion(obscoreVersion);
        editedCredentials.setIvoaSchemaName(ivoaSchemaName);

        if (credentialsInEditing == null) {
            // New credentials
            config.addCredentials(user.getUsername(), editedCredentials);
        } else {
            // Existing credentials
            config.updateCredentials(user.getUsername(), editedCredentials, credentialsInEditing);
        }
    }

    public List<String> getTapSchemaVersions() {
        return tapSchemaVersions;
    }

    public List<String> getObscoreVersions() {
        return obscoreVersions;
    }

    public String getLabel() {
        return label;
    }

    public void setLabel(String label) {
        this.label = label;
    }

    public boolean isSeparateCredentials() {
        return separateCredentials;
    }

    public void setSeparateCredentials(boolean separateCredentials) {
        this.separateCredentials = separateCredentials;
    }

    public boolean isCredentialsDialogOpened() {
        return credentialsDialogOpened;
    }

    public void setCredentialsDialogOpened(boolean credentialsDialogOpened) {
        this.credentialsDialogOpened = credentialsDialogOpened;
    }

    public Credentials getSourceCredentials() {
        return sourceCredentials;
    }

    public void setSourceCredentials(Credentials sourceCredentials) {
        this.sourceCredentials = sourceCredentials;
    }

    public Credentials getTapSchemaCredentials() {
        return tapSchemaCredentials;
    }

    public void setTapSchemaCredentials(Credentials tapSchemaCredentials) {
        this.tapSchemaCredentials = tapSchemaCredentials;
    }

    public String getTapSchemaName() {
        return tapSchemaName;
    }

    public void setTapSchemaName(String tapSchemaName) {
        this.tapSchemaName = tapSchemaName;
    }

    public String getTapSchemaVersion() {
        return tapSchemaVersion;
    }

    public void setTapSchemaVersion(String tapSchemaVersion) {
        this.tapSchemaVersion = tapSchemaVersion;
    }

    public boolean isHasObscore() {
        return hasObscore;
    }

    public void setHasObscore(boolean hasObscore) {
        this.hasObscore = hasObscore;
    }

    public String getObscoreVersion() {
        return obscoreVersion;
    }

    public void setObscoreVersion(String obscoreVersion) {
        this.obscoreVersion = obscoreVersion;
    }

    public String getIvoaSchemaName() {
        return ivoaSchemaName;
    }

    public void setIvoaSchemaName(String ivoaSchemaName) {
        this.ivoaSchemaName = ivoaSchemaName;
    }
}
