/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import it.inaf.ia2.tsm.model.ColumnModel;
import it.inaf.ia2.tsm.model.TableModel;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Represents an object that is mapped on a table of the TAP_SCHEMA.
 *
 * A {@code TapSchemaEntity} has some properties that correspond to columns of
 * the table represented by the {@code TapSchemaEntity}.
 * <p>
 * Property value can be changed but the original value has to be maintained
 * until the {@link #save()} method is called.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public abstract class TapSchemaEntity implements Serializable {

    private static final long serialVersionUID = 5515596028279668709L;
    private static final Logger LOG = LoggerFactory.getLogger(TapSchemaEntity.class);

    private Map<String, Object> metadata;
    private Map<String, EntityProperty> properties;
    protected TapSchema tapSchema;
    private TableModel tableModel;

    /**
     * Only for serialization.
     */
    protected TapSchemaEntity() {
    }

    /**
     * Default constructor.
     *
     * @param tapSchema the {@code TapSchema} owning this entity.
     * @param tableModel the {@code TableModel} defining this entity (for
     * example, if the instance of this entity is a {@link Column}, the
     * {@code TableModel} represents the TAP_SCHEMA {@code columns} table).
     * @param metadata a map containing the property values read from the
     * database metadata or information_schema (this properties can't be edited
     * or have always a suggested value).
     */
    public TapSchemaEntity(TapSchema tapSchema, TableModel tableModel, Map<String, Object> metadata) {
        this.tapSchema = tapSchema;
        this.tableModel = tableModel;
        this.metadata = metadata;
        this.properties = new HashMap<>();
        fillProperties();
    }

    /**
     * Initializes the properties owned by this entity, retrieving them from the
     * {@code metadata} map.
     */
    private void fillProperties() {
        for (ColumnModel propModel : tableModel.getColumns()) {
            Object defaultValue = null;
            if (propModel.getLoaderKey() != null) {
                defaultValue = metadata.get(propModel.getLoaderKey());

                if (defaultValue != null) {
                    // Special case for boolean to integer conversion
                    Class javaType = propModel.getJavaType();
                    if (javaType != defaultValue.getClass()) {
                        if (defaultValue.getClass() == Boolean.class && javaType == Integer.class) {
                            defaultValue = ((Boolean) defaultValue) ? 1 : 0;
                        } else {
                            throw new UnsupportedOperationException("Unable to convert " + defaultValue.getClass().getCanonicalName() + " into " + propModel.getType());
                        }
                    }
                }
            }
            EntityProperty ep = new EntityProperty(propModel, defaultValue);
            this.properties.put(propModel.getName(), ep);
        }
    }

    /**
     * Returns a property value that has been read from the database metadata
     * (or information_schema), given the name of the property owned by this
     * entity.
     *
     * @param key the name of a property owned by this entity.
     * @return the value of the property read from the database metadata, null
     * if no value for the given property name has been retrieved.
     */
    public Object getMetadata(String key) {
        return metadata.get(key);
    }

    /**
     * Initializes the value of a property (stores the original value).
     *
     * @param <T> a class compatible with the definition for the given column.
     * @param key the name of the property.
     * @param value the initial value for the property.
     */
    public <T> void initProperty(String key, T value) {
        properties.get(key).init(value);
    }

    /**
     * Returns true if one or more property values is changed (the current value
     * is different from the original value).
     *
     * @return true if one or more properties owned by this entity have been
     * changed, false otherwise.
     */
    public boolean isChanged() {
        for (EntityProperty property : properties.values()) {
            if (property.isChanged()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns the value of a property owned by this entity, given its name.
     *
     * @param key the name of the property.
     * @return the value of the property.
     */
    public Object getValue(String key) {
        return properties.get(key).getValue();
    }

    /**
     * Returns the Java class type of a property owned by this entity, given its
     * name.
     *
     * @param key key the name of the property.
     * @return the Java class type of the property.
     */
    public Class getPropertyType(String key) {
        return properties.get(key).getType();
    }

    /**
     * Returns a property owned by this entity, given its name.
     *
     * @param key the name of the property.
     * @return an {@link EntityProperty}
     */
    public EntityProperty getProperty(String key) {
        return properties.get(key);
    }

    /**
     * Returns the current value of the property (the last value set).
     */
    public <T> T getValue(String key, Class<T> type) {
        return (T) properties.get(key).getValue();
    }

    /**
     * Sets a new value to a property owned by this entity.
     *
     * @param key the name of the property.
     * @param value the new value of the property.
     */
    public void setValue(String key, Object value) {
        properties.get(key).setValue(value);
    }

    /**
     * Retrieve the original/initial value of the property.
     *
     * @param key the name of the property (the name of the table column).
     * @param type the class of the property value.
     * @return the value with which the property has been initialized.
     */
    public <T> T getOriginalValue(String key, Class<T> type) {
        return (T) properties.get(key).getOriginalValue();
    }

    /**
     * Returns true the value of property the key of which is passed as
     * parameter is changed (the current value is different from the original
     * value).
     */
    public boolean isChanged(String key) {
        return properties.get(key).isChanged();
    }

    /**
     * Retrieves a list of all properties names (the names of the table columns)
     * owned by this entity.
     */
    public List<String> getPropertiesKeys() {
        return new ArrayList<>(properties.keySet());
    }

    /**
     * Marks this entity as saved (all original values are set equals to their
     * current values).
     */
    public void save() {
        for (EntityProperty p : properties.values()) {
            p.save();
        }
    }

    /**
     * Returns the {@code TableModel} defining this entity (for example, if the
     * instance of this entity is a {@link Column}, the {@code TableModel}
     * represents the TAP_SCHEMA {@code columns} table).
     */
    public TableModel getTableModel() {
        return tableModel;
    }
}
