/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import it.inaf.ia2.tsm.webapp.xmlconfig.Configuration;
import it.inaf.ia2.tsm.webapp.xmlconfig.UserConfiguration;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;
import javax.annotation.PostConstruct;
import javax.enterprise.context.ApplicationScoped;
import javax.faces.context.FacesContext;
import javax.inject.Named;
import javax.xml.bind.JAXB;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named("config")
@ApplicationScoped
public class ConfigurationData {

    private static final Logger LOG = LoggerFactory.getLogger(ConfigurationData.class);

    private File configFile;
    private Configuration configuration;

    @PostConstruct
    public void init() {
        try {
            Properties prop = new Properties();
            try (InputStream in = getClass().getClassLoader().getResourceAsStream("webapp.properties")) {
                prop.load(in);
            }
            configFile = new File(prop.getProperty("config_file_path"));

            if (!configFile.exists()) {

                LOG.debug("Configuration file doesn't exist: creating a new one at " + configFile.getAbsolutePath());

                configFile.getParentFile().mkdirs();
                configFile.createNewFile();
                configuration = new Configuration();
                updateConfigurationFile();

            } else {

                // Configuration file exists, simply unmarshal it
                configuration = JAXB.unmarshal(configFile, Configuration.class);
            }
        } catch (IOException e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    public Configuration getData() {
        return configuration;
    }

    public synchronized List<UserConfiguration> cloneUsersConfiguration() {
        // JAXB is exploited for doing deep copy.
        return JAXB.unmarshal(configFile, Configuration.class).getUsers();
    }

    public void updateConfigurationFile() throws IOException {
        JAXB.marshal(configuration, configFile);
    }

    public synchronized void addUser(UserConfiguration user) throws IOException {
        configuration.getUsers().add(user);
        updateConfigurationFile();
    }

    public synchronized void updateUsersList(List<UserConfiguration> users) throws IOException {
        configuration.getUsers().clear();
        for (UserConfiguration user : users) {
            configuration.getUsers().add(user);
        }
        updateConfigurationFile();
    }

    public String getRestPath() {
        return FacesContext.getCurrentInstance().getExternalContext().getRequestContextPath() + "/faces/rest";
    }

    public String getVersion() {
        return Version.NUMBER;
    }
}
