/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.api;

import it.inaf.ia2.tsm.api.contract.TapSchema;
import it.inaf.ia2.tsm.api.contract.TapSchemaEntity;
import it.inaf.ia2.tsm.api.contract.TapSchemaVersion;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The main implementation of {@link TapSchemaEntity}.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public abstract class TapSchemaEntityImpl implements TapSchemaEntity, Serializable {
    
    private static final long serialVersionUID = 5515596028279668709L;
    private static final Logger log = LoggerFactory.getLogger(TapSchemaEntityImpl.class);
    
    private final Map<String, EntityProperty> properties;
    private final String tapSchemaEntityTable;
    protected DBWrapper dbWrapper;
    protected TapSchema tapSchema;
    
    protected TapSchemaEntityImpl() {
        // for serialization
        properties = new HashMap<>();
        tapSchemaEntityTable = TSMUtil.getTapSchemaTableNameFromEntity((this));
    }
    
    public TapSchemaEntityImpl(DBWrapper dbWrapper, TapSchema tapSchema) {
        this();
        this.dbWrapper = dbWrapper;
        this.tapSchema = tapSchema;
    }
    
    protected void addProperty(String key, EntityProperty property) {
        if (EntityPropertyInfo.getEntityPropertyInfo(tapSchemaEntityTable, key).acceptVersion(tapSchema.getVersion())) {
            properties.put(key, property);
        }
    }
    
    protected DBWrapper getDBWrapper() {
        return dbWrapper;
    }
    
    @Override
    public <T> void initProperty(String key, T value) {
        properties.get(key).init(value);
    }
    
    protected TapSchemaVersion getVersion() {
        return tapSchema.getVersion();
    }
    
    @Override
    public boolean isChanged() {
        for (EntityProperty property : properties.values()) {
            if (property.isChanged()) {
                return true;
            }
        }
        return false;
    }
    
    @Override
    public <T> T getValue(String key, Class<T> type) {
        if (!EntityPropertyInfo.getEntityPropertyInfo(tapSchemaEntityTable, key).acceptVersion(tapSchema.getVersion())) {
            throw TSMUtil.getUnsupportedOperationException(getVersion(), key + " property");
        }
        return properties.get(key).getValue(type);
    }
    
    public <T> void setValue(String key, T value) {
        EntityProperty property = properties.get(key);
        if (property instanceof EditableProperty) {
            if (!EntityPropertyInfo.getEntityPropertyInfo(tapSchemaEntityTable, key).acceptVersion(tapSchema.getVersion())) {
                throw TSMUtil.getUnsupportedOperationException(getVersion(), key + " property");
            }
            ((EditableProperty) property).setValue(value);
        } else {
            throw new IllegalArgumentException("Property " + key + " isn't editable");
        }
    }
    
    @Override
    public <T> T getOriginalValue(String key, Class<T> type) {
        EntityProperty property = properties.get(key);
        if (property instanceof EditableProperty) {
            return ((EditableProperty<T>) property).getOriginalValue(type);
        }
        throw new IllegalArgumentException("Property " + key + " hasn't original value");
    }
    
    @Override
    public boolean isChanged(String key) {
        return properties.get(key).isChanged();
    }
    
    @Override
    public List<String> getPropertiesKeys() {
        return new ArrayList<>(properties.keySet());
    }
    
    @Override
    public void save() {
        for (EntityProperty p : properties.values()) {
            if (p instanceof EditableProperty) {
                EditableProperty property = (EditableProperty) p;
                property.save();
            } else {
                FixedEntityProperty property = (FixedEntityProperty) p;
                property.setChanged(false);
            }
        }
    }
    
    @Override
    public <T> void amendProperty(String key, T value) {
        FixedEntityProperty prop = (FixedEntityProperty) properties.get(key);
        prop.init(value);
        prop.setChanged(true);
    }
}
