/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.api;

import it.inaf.ia2.tsm.api.contract.DatabaseType;
import it.inaf.ia2.tsm.api.contract.TapSchema;
import it.inaf.ia2.tsm.api.contract.TapSchemaEntity;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Prepares a {@code SELECT} SQL query for a given {@link TapSchemaEntity} and a
 * given {@link TapSchema}.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public abstract class SelectQueryBuilder {

    private static final Logger log = LoggerFactory.getLogger(SelectQueryBuilder.class);

    private final TapSchema tapSchema;
    private final String query;
    private final List<EntityPropertyInfo> addedProperties;

    protected SelectQueryBuilder(DatabaseType dbType, TapSchema tapSchema, String tapSchemaTableName) {

        this.tapSchema = tapSchema;

        StringBuilder querySb = new StringBuilder("SELECT ");

        addedProperties = new ArrayList<>();

        boolean first = true;
        for (EntityPropertyInfo propertyInfo : EntityPropertyInfo.getEntityPropertiesInfo(tapSchemaTableName)) {
            if (propertyInfo.acceptVersion(tapSchema.getVersion())) {
                if (!first) {
                    querySb.append(", ");
                }
                querySb.append(propertyInfo.getPropertyKey());
                addedProperties.add(propertyInfo);
                first = false;
            }
        }

        querySb.append(" FROM ");

        querySb.append(TSMUtil.escapeName(tapSchema.getName(), dbType));
        querySb.append(".");
        querySb.append(TSMUtil.escapeName(tapSchemaTableName, dbType));

        query = querySb.toString();
    }

    protected abstract TapSchemaEntity getEntity(ResultSet rs) throws SQLException;

    protected void executeQuery(Connection connection) throws SQLException {

        log.debug("Executing query {}", query);

        try (Statement statement = connection.createStatement();
                ResultSet rs = statement.executeQuery(query)) {

            while (rs.next()) {

                TapSchemaEntity entity = getEntity(rs);

                if (entity != null) {
                    for (EntityPropertyInfo property : addedProperties) {
                        String key = property.getPropertyKey();
                        Class type = property.getPropertyType();
                        Object value = TSMUtil.getObject(rs, key, type);
                        if (property.isUpdatable()) {
                            entity.initProperty(key, value);
                        } else {
                            Object correctValue = entity.getValue(key, type);
                            boolean changed = (correctValue == null && value != null) || (correctValue != null && !correctValue.equals(value));
                            if (changed) {

                                entity.amendProperty(key, correctValue);

                                InconsistentValue inconsistentValue = new InconsistentValue(
                                        TSMUtil.getNaturalLangueName(entity),
                                        TSMUtil.getName(entity),
                                        key,
                                        value,
                                        correctValue
                                );

                                tapSchema.getConsistencyChecks().addInconsistency(inconsistentValue);

                                //throw new InconsistentTapSchemaException(debugInfo);
                            }
                        }
                    }
                }
            }
        }
    }

    public String getQuery() {
        return query;
    }
}
