/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.api;

import it.inaf.ia2.tsm.api.contract.DatabaseType;
import java.io.Serializable;
import javax.xml.bind.annotation.XmlAttribute;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Representation of credentials used to connect to a RDBMS. It can be
 * serialized in XML using the JAXB standard.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class Credentials implements Serializable {

    private static final long serialVersionUID = 1153912575502196261L;
    private static final Logger log = LoggerFactory.getLogger(Credentials.class);

    private String hostname;
    private int port;
    private String username;
    private String password;
    private String database;

    private DatabaseType databaseType;

    public Credentials() {
        this(DatabaseType.MYSQL);
    }

    public Credentials(DatabaseType dbType) {
        this.databaseType = dbType;
        this.setDefaults();
    }

    /**
     * The name of the server that hosts the RDBMS.
     */
    @XmlAttribute(name = "hostname", required = true)
    public String getHostname() {
        return this.hostname;
    }

    public void setHostname(String hostname) {
        this.hostname = hostname;
    }

    /**
     * The RDBMS port number.
     */
    @XmlAttribute(name = "port", required = true)
    public int getPort() {
        return this.port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    @XmlAttribute(name = "username", required = true)
    public String getUsername() {
        return this.username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    @XmlAttribute(name = "password", required = true)
    public String getPassword() {
        return this.password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    /**
     * The kind of RDBMS used.
     */
    @XmlAttribute(name = "database-type", required = true)
    public DatabaseType getDatabaseType() {
        return databaseType;
    }

    public void setDatabaseType(DatabaseType databaseType) {
        this.databaseType = databaseType;
    }

    /**
     * <strong>Only for POSTGRES</strong>. The database to use in the
     * connection. The default value is "postgres".
     */
    @XmlAttribute(name = "database")
    public String getDatabase() {
        return database;
    }

    /**
     * <strong>Only for POSTGRES</strong>. The database to use in the
     * connection.
     */
    public void setDatabase(String database) {
        this.database = database;
    }

    /**
     * Set default values for <code>databaseType</code>, <code>database</code>
     * and <code>port</code> properties.
     */
    public final void setDefaults() {
        if (databaseType == DatabaseType.MYSQL) {
            this.port = 3306;
            this.database = null;
        } else if (databaseType == DatabaseType.POSTGRES) {
            this.port = 5432;
            this.database = "postgres";
        }
    }

    /**
     * The class name of the RDBMS driver.
     */
    public String getDatabaseDriverClass() {
        switch (getDatabaseType()) {
            case MYSQL:
                return "com.mysql.jdbc.Driver";
            case POSTGRES:
                return "org.postgresql.Driver";
        }
        return null;
    }

    @Override
    public String toString() {
        return String.format("[%s] type=%s, hostname=%s, port=%s, username=%s, password=%s, database=%s",
                Credentials.class.getCanonicalName(),
                databaseType, hostname, port, username, password, database);
    }
}
