/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import it.inaf.ia2.tsm.TapSchema;
import it.inaf.ia2.tsm.TapSchemaSettings;
import it.inaf.ia2.tsm.datalayer.DBBroker;
import it.inaf.ia2.tsm.datalayer.DBBrokerFactory;
import it.inaf.ia2.tsm.datalayer.DBWrapper;
import it.inaf.ia2.tsm.datalayer.DataTypeMode;
import it.inaf.ia2.tsm.model.SchemaModels;
import it.inaf.ia2.tsm.webapp.env.CustomPartialResponseWriter;
import it.inaf.ia2.tsm.webapp.xmlconfig.JoinedCredentials;
import it.inaf.ia2.tsm.webapp.xmlconfig.SeparatedCredentials;
import it.inaf.ia2.tsm.webapp.xmlconfig.TapCredentials;
import java.io.Serializable;
import java.sql.SQLException;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.deltaspike.core.api.scope.WindowScoped;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named("tapSchemaLoader")
@WindowScoped
public class TapSchemaLoader implements Serializable {

    private static final long serialVersionUID = 3203810003976020854L;

    private static final Logger LOG = LoggerFactory.getLogger(TapSchemaLoader.class);

    @Inject
    private ConsistencyChecksBean consistencyChecksBean;

    @Inject
    TapSchemaEditingBean tapSchemaEditingBean;

    private TapCredentials tapCredentials;
    private DBWrapper dbWrapper;

    private boolean loading;
    private TapSchema loadedTapSchema;
    private String loadingError;

    public void tryLoadingTapSchema(TapCredentials tapCredentials) throws SQLException {
        this.tapCredentials = tapCredentials;

        if (tapCredentials instanceof JoinedCredentials) {
            JoinedCredentials joinedCredentials = (JoinedCredentials) tapCredentials;
            dbWrapper = new DBWrapper(joinedCredentials.getCredentials());
        } else {
            SeparatedCredentials separatedCredentials = (SeparatedCredentials) tapCredentials;
            dbWrapper = new DBWrapper(separatedCredentials.getSourceCredentials(), separatedCredentials.getTapSchemaCredentials());
        }

        // Testing connections
        dbWrapper.testConnections();

        // Searching for TAP_SCHEMA name
        DataTypeMode dataTypeMode = SchemaModels.getTapSchemaModel(tapCredentials.getTapSchemaVersion()).getDataTypeMode();
        DBBroker broker = DBBrokerFactory.getDBBroker(dbWrapper.getTapSchemaDataSourceWrapper(), dataTypeMode);
        boolean tapSchemaExists = false;
        for (String schemaName : broker.getAllSchemaNames()) {
            if (schemaName.equals(tapCredentials.getTapSchemaName())) {
                tapSchemaExists = true;
                break;
            }
        }

        CustomPartialResponseWriter.getCurrentInstance().addCustomJSUpdate("tap_schema_existence", String.valueOf(tapSchemaExists));

        if (tapSchemaExists) {
            edit();
        }
        // Otherwise create TAP_SCHEMA only if user press Ok on confirmation dialog
    }

    private TapSchemaSettings getTapSchemaSettings(TapCredentials tapCredentials) {
        TapSchemaSettings settings = new TapSchemaSettings();
        settings.setTapSchemaName(tapCredentials.getTapSchemaName());
        settings.setTapSchemaVersion(tapCredentials.getTapSchemaVersion());
        settings.setHasObscore(tapCredentials.isHasObscore());
        settings.setObscoreVersion(tapCredentials.getObscoreVersion());
        return settings;
    }

    public void edit() {

        loadedTapSchema = null;
        loading = true;
        loadingError = null;

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {

                    loadedTapSchema = new TapSchema(dbWrapper, getTapSchemaSettings(tapCredentials), true);
                } catch (Throwable e) {
                    LOG.error("Exception caught", e);
                    loadingError = e.getMessage();
                    if (loadingError == null) {
                        loadingError = e.getClass().getCanonicalName();
                    }
                }
                loading = false;
            }
        }).start();
    }

    public void create() {

        loadedTapSchema = null;
        loading = true;
        loadingError = null;

        new Thread(new Runnable() {
            @Override
            public void run() {
                try {
                    loadedTapSchema = new TapSchema(dbWrapper, getTapSchemaSettings(tapCredentials), false);
                } catch (Throwable e) {
                    LOG.error("Exception caught", e);
                    if (e.getMessage() != null) {
                        loadingError = e.getMessage();
                    } else {
                        loadingError = e.getClass().getCanonicalName();
                        loadedTapSchema = null;
                    }
                }
                loading = false;
            }
        }).start();
    }

    public TapCredentials getTapCredentials() {
        return tapCredentials;
    }

    public boolean isLoading() {
        return loading;
    }

    public TapSchema getLoadedTapSchema() {
        return loadedTapSchema;
    }

    public String getLoadingError() {
        return loadingError;
    }

    public String openLoaded() {
        if (loadedTapSchema.getConsistencyChecks().isInconsistent()) {
            consistencyChecksBean.setTapSchema(loadedTapSchema);
            return "consistencyChecks.xhtml?faces-redirect=true";
        } else {
            tapSchemaEditingBean.setTapSchema(loadedTapSchema);
            return "tapSchemaEditing.xhtml?faces-redirect=true";
        }
    }
}
