/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import it.inaf.ia2.tsm.webapp.xmlconfig.TapCredentials;
import it.inaf.ia2.tsm.webapp.xmlconfig.UCDConfiguration;
import it.inaf.ia2.tsm.webapp.xmlconfig.UCDListConfiguration;
import it.inaf.ia2.tsm.webapp.xmlconfig.UsersConfiguration;
import it.inaf.ia2.tsm.webapp.xmlconfig.UserConfiguration;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import javax.annotation.PostConstruct;
import javax.enterprise.context.ApplicationScoped;
import javax.faces.context.FacesContext;
import javax.inject.Named;
import javax.xml.bind.JAXB;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named("config")
@ApplicationScoped
public class ConfigurationManager {

    private static final Logger LOG = LoggerFactory.getLogger(ConfigurationManager.class);

    private static final String USERS_CONFIG_FILE_NAME = "users.xml";
    private static final String UCD_CONFIG_FILE_NAME = "ucd.xml";

    private File configDirectory;
    private File usersConfigFile;
    private File ucdConfigFile;

    private UsersConfiguration usersConfig;
    private UCDListConfiguration ucdListConfig;

    @PostConstruct
    public void init() {
        try {
            Properties prop = new Properties();
            try (InputStream in = ConfigurationManager.class.getClassLoader().getResourceAsStream("webapp.properties")) {
                prop.load(in);
            }
            configDirectory = new File(prop.getProperty("config_directory"));
            usersConfigFile = configDirectory.toPath().resolve(USERS_CONFIG_FILE_NAME).toFile();
            ucdConfigFile = configDirectory.toPath().resolve(UCD_CONFIG_FILE_NAME).toFile();

            if (!configDirectory.exists()) {
                configDirectory.mkdirs();
            }

            if (!usersConfigFile.exists()) {
                usersConfigFile.createNewFile();
                usersConfig = new UsersConfiguration();
                updateUsersConfigurationFile();
            } else {
                usersConfig = JAXB.unmarshal(usersConfigFile, UsersConfiguration.class);
            }

            if (!ucdConfigFile.exists()) {
                ucdConfigFile.createNewFile();
                ucdListConfig = new UCDListConfiguration();
                updateUCDConfigurationFile();
            } else {
                ucdListConfig = JAXB.unmarshal(ucdConfigFile, UCDListConfiguration.class);
            }
        } catch (IOException e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    public List<UserConfiguration> getUsersConfiguration() {
        return Collections.unmodifiableList(usersConfig.getUsers());
    }

    public UserConfiguration getUserConfiguration(String username) {
        for (UserConfiguration user : usersConfig.getUsers()) {
            if (username.equals(user.getUsername())) {
                return user;
            }
        }
        return null;
    }

    public List<UCDConfiguration> getUCDConfiguration() {
        return Collections.unmodifiableList(ucdListConfig.getUCDList());
    }

    private void updateUsersConfigurationFile() {
        JAXB.marshal(usersConfig, usersConfigFile);
    }

    private void updateUCDConfigurationFile() {
        JAXB.marshal(ucdListConfig, ucdConfigFile);
    }

    /**
     * Add a new user to the list, but only if none of the existing users have
     * the same username.
     *
     * @return true if the user has been added to the list, false otherwise.
     */
    public synchronized boolean addUser(UserConfiguration user) {
        if (user.getUsername() == null) {
            return false;
        }

        for (UserConfiguration u : usersConfig.getUsers()) {
            if (u.getUsername().equals(user.getUsername())) {
                return false;
            }
        }

        usersConfig.getUsers().add(user);
        updateUsersConfigurationFile();
        return true;
    }

    public synchronized boolean deleteUser(String username) {
        Iterator<UserConfiguration> ite = usersConfig.getUsers().iterator();
        while (ite.hasNext()) {
            UserConfiguration userConfig = ite.next();
            if (userConfig.getUsername().equals(username)) {
                ite.remove();
                updateUsersConfigurationFile();
                return true;
            }
        }
        return false;
    }

    public synchronized boolean editUser(String username, String role, String password) {

        for (UserConfiguration u : usersConfig.getUsers()) {
            if (u.getUsername().equals(username)) {
                u.setRole(role);
                u.setPassword(password);
                updateUsersConfigurationFile();
                return true;
            }
        }

        return false;
    }

    /**
     * Add a new UCD to the list, but only if the word has not already been
     * entered.
     *
     * @return true if the UCD has been added to the list, false otherwise.
     */
    public synchronized boolean addUCD(UCDConfiguration ucdConfig) {
        for (UCDConfiguration ucd : ucdListConfig.getUCDList()) {
            if (ucd.getWord().equals(ucdConfig.getWord())) {
                return false;
            }
        }

        ucdListConfig.getUCDList().add(ucdConfig);
        updateUCDConfigurationFile();
        return true;
    }

    /**
     * @return true if the UCD has been deleted from the list, false otherwise.
     */
    public synchronized boolean deleteUCD(UCDConfiguration ucdConfig) {
        Iterator<UCDConfiguration> ite = ucdListConfig.getUCDList().iterator();
        while (ite.hasNext()) {
            UCDConfiguration ucd = ite.next();
            if (ucd.getWord().equals(ucdConfig.getWord())) {
                ite.remove();
                updateUCDConfigurationFile();
                return true;
            }
        }
        return false;
    }

    public List<TapCredentials> getCredentials(String username) {
        List<TapCredentials> credentials = getUserCredentials(username);
        if (credentials == null) {
            return null;
        }
        return Collections.unmodifiableList(credentials);
    }

    private List<TapCredentials> getUserCredentials(String username) {
        for (UserConfiguration user : usersConfig.getUsers()) {
            if (user.getUsername().equals(username)) {
                return user.getCredentialsInfo();
            }
        }
        return null;
    }

    /**
     * Add a new credentials to the list, but only if the credentials has not
     * already been inserted.
     *
     * @return true if the credentials has been added to the list, false
     * otherwise.
     */
    public synchronized boolean addCredentials(String username, TapCredentials credentials) {
        List<TapCredentials> userCredentials = getUserCredentials(username);
        if (userCredentials != null) {
            userCredentials.add(credentials);
            updateUsersConfigurationFile();
            return true;
        }
        return false;
    }

    /**
     * @return true if the credentials has been deleted from the list, false
     * otherwise.
     */
    public synchronized boolean deleteCredentials(String username, int credentialsIndex) {
        List<TapCredentials> userCredentials = getUserCredentials(username);
        if (userCredentials != null && userCredentials.size() > credentialsIndex) {
            userCredentials.remove(credentialsIndex);
            updateUsersConfigurationFile();
            return true;
        }
        return false;
    }

    public synchronized boolean updateCredentials(String username, TapCredentials updatedCredentials, int index) {
        List<TapCredentials> userCredentials = getUserCredentials(username);
        if (userCredentials != null && userCredentials.size() > index) {
            userCredentials.set(index, updatedCredentials);
            updateUsersConfigurationFile();
            return true;
        }
        return false;
    }

    public String getRestPath() {
        return FacesContext.getCurrentInstance().getExternalContext().getRequestContextPath() + "/faces/rest";
    }

    public String getVersion() {
        return Version.NUMBER;
    }
}
