/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;

/**
 * The main implementation of {@link Key}.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class Key extends TapSchemaEntity implements Serializable {

    public static final String ID_KEY = "key_id";
    public static final String FROM_TABLE_KEY = "from_table";
    public static final String TARGET_TABLE_KEY = "target_table";

    private static final long serialVersionUID = -8783695875831579336L;

    private List<KeyColumn> keyColumns;
    private boolean visible;

    private String fromSchema;
    private String fromTable;
    private String targetSchema;
    private String targetTable;

    private Key() {
        super();
    }

    public Key(TapSchema tapSchema, Map<String, Object> metadata) {
        super(tapSchema, tapSchema.getTableModel(TapSchema.KEYS_TABLE), metadata);

        keyColumns = new ArrayList<>();
        visible = false;

        String[] fromSplit = ((String) metadata.get(FROM_TABLE_KEY)).split(Pattern.quote("."));
        String[] targetSplit = ((String) metadata.get(TARGET_TABLE_KEY)).split(Pattern.quote("."));

        fromSchema = fromSplit[0];
        fromTable = fromSplit[1];
        targetSchema = targetSplit[0];
        targetTable = targetSplit[1];
    }

    public boolean isVisible() {
        return visible;
    }

    protected void setVisible(boolean visible) {
        this.visible = visible;
    }

    public String getId() {
        return getValue(ID_KEY, String.class);
    }

    public void setId(String id) {
        setValue(ID_KEY, id);
        for (KeyColumn keyColumn : keyColumns) {
            keyColumn.setKeyId(id);
        }
    }

    public List<KeyColumn> getKeyColumns() {
        return Collections.unmodifiableList(keyColumns);
    }

    public String getFromSchemaName() {
        return fromSchema;
    }

    public String getFromTableSimpleName() {
        return fromTable;
    }

    public String getFromTableCompleteName() {
        return getValue(FROM_TABLE_KEY, String.class);
    }

    public String getTargetSchemaName() {
        return targetSchema;
    }

    public String getTargetTableSimpleName() {
        return targetTable;
    }

    public String getTargetTableCompleteName() {
        return getValue(TARGET_TABLE_KEY, String.class);
    }

    public KeyColumn addKeyColumn(String fromColumn, String targetColumn) {
        Map<String, Object> keyColumnMetadata = new HashMap<>();
        keyColumnMetadata.put(KeyColumn.FROM_COLUMN_KEY, fromColumn);
        keyColumnMetadata.put(KeyColumn.TARGET_COLUMN_KEY, targetColumn);
        KeyColumn keyColumn = new KeyColumn(tapSchema, this, keyColumnMetadata);
        keyColumns.add(keyColumn);
        return keyColumn;
    }

    @Override
    public void save() {
        if (!isVisible()) {
            initProperty(ID_KEY, null);
        }
        super.save();
        for (KeyColumn keyColumn : keyColumns) {
            keyColumn.save();
        }
    }

    @Override
    public String toString() {
        return TSMUtil.printKeyInfo(this);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        for (KeyColumn keyColumn : keyColumns) {
            hash = 23 * hash + Objects.hashCode(keyColumn);
        }
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (Key.class != obj.getClass()) {
            return false;
        }

        final Key other = (Key) obj;
        List<KeyColumn> otherKeyColumns = other.getKeyColumns();

        // Comparing each key column
        if (keyColumns.size() != otherKeyColumns.size()) {
            return false;
        }
        for (int i = 0; i < keyColumns.size(); i++) {
            if (!keyColumns.get(i).equals(otherKeyColumns.get(i))) {
                return false;
            }
        }

        return true;
    }
}
