/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class Column extends ChildEntity<Table> {

    public final static String TABLE_NAME_KEY = "table_name";
    public final static String COLUMN_NAME_KEY = "column_name";
    public final static String DATATYPE_KEY = "datatype";
    public final static String SIZE_KEY = "size";
    public final static String ARRAYSIZE_KEY = "arraysize";
    public final static String INDEXED_KEY = "indexed";
    public final static String PRIMARY_KEY = "primary_key";
    public final static String PRINCIPAL_KEY = "principal";
    public final static String STD_KEY = "std";
    public final static String UTYPE_KEY = "utype";
    public final static String UCD_KEY = "ucd";
    public final static String UNIT_KEY = "unit";

    private static final long serialVersionUID = 9175956487892235521L;
    private static final Logger LOG = LoggerFactory.getLogger(Column.class);

    private boolean foreignKeySearched;
    private Key foreignKey;

    private Table parentTable;

    private Column() {
        // for serialization
        super();
    }

    protected Column(TapSchema tapSchema, Table table, String columnName) {
        super(tapSchema, tapSchema.getTableModel(TapSchema.COLUMNS_TABLE), table.getColumnMetadata(columnName));
        parentTable = table;
        setStatus(Status.LOADED);
    }

    public Key getForeignKey() {
        if (!foreignKeySearched) { // lazy loading (but the foreignKey value can be null, so we use this boolean)

            String fromSchemaName = parentTable.getParent().getName();
            String fromTableName = parentTable.getName();
            String fromColumnName = getName();

            ext:
            for (Key key : tapSchema.getAllKeys()) {
                if (key.getFromSchemaName().equals(fromSchemaName) && key.getFromTableSimpleName().equals(fromTableName)) {
                    for (KeyColumn keyColumn : key.getKeyColumns()) {
                        if (keyColumn.getFromColumn().equals(fromColumnName)) {
                            foreignKey = key;
                            break ext;
                        }
                    }
                }
            }
        }

        return foreignKey;
    }

    public String getTableCompleteName() {
        return getValue(TABLE_NAME_KEY, String.class);
    }

    @Override
    public String getName() {
        return getValue(COLUMN_NAME_KEY, String.class);
    }

    public boolean getIndexed() {
        EntityProperty indexedProp = getProperty(INDEXED_KEY);
        if (indexedProp.getType() == Boolean.class) {
            return getValue(INDEXED_KEY, Boolean.class);
        } else if (indexedProp.getType() == Integer.class) {
            return getValue(INDEXED_KEY, Integer.class) == 1;
        } else {
            throw new RuntimeException("Indexed property cannot be an instance of " + indexedProp.getType());
        }
    }

    /**
     * This information is not stored into the TAP_SCHEMA, so it will be
     * reloaded from the source schema each time.
     */
    public boolean isPrimaryKey() {
        return (boolean) getMetadata(PRIMARY_KEY);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 29 * hash + Objects.hashCode(getTableCompleteName());
        hash = 29 * hash + Objects.hashCode(getName());
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Column other = (Column) obj;
        if (!Objects.equals(this.getTableCompleteName(), other.getTableCompleteName())) {
            return false;
        }
        return Objects.equals(this.getName(), other.getName());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Table getParent() {
        return parentTable;
    }
}
