/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import java.io.Serializable;
import java.util.Objects;

/**
 * Models the reference to a column in the database during the consistency
 * checking phase (it is used to display information about columns not already
 * loaded/added into the TAP_SCHEMA or to represent a column which doesn't exist
 * yet).
 *
 * @see it.inaf.ia2.tsm.ConsistencyChecks
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class ColumnHolder implements Serializable {

    private static final long serialVersionUID = 5837270912530393588L;

    private String schemaName;
    private String tableName;
    private String columnName;

    private ColumnHolder() {
    }

    public ColumnHolder(String schemaName, String tableName, String columnName) {
        this.schemaName = schemaName;
        this.tableName = tableName;
        this.columnName = columnName;
    }

    public ColumnHolder(Column column) {
        this.schemaName = column.getParent().getParent().getName();
        this.tableName = column.getParent().getName();
        this.columnName = column.getName();
    }

    public String getSchemaName() {
        return schemaName;
    }

    public void setSchemaName(String schemaName) {
        this.schemaName = schemaName;
    }

    public String getTableName() {
        return tableName;
    }

    public void setTableName(String tableName) {
        this.tableName = tableName;
    }

    public String getColumnName() {
        return columnName;
    }

    public void setColumnName(String columnName) {
        this.columnName = columnName;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 71 * hash + Objects.hashCode(this.schemaName);
        hash = 71 * hash + Objects.hashCode(this.tableName);
        hash = 71 * hash + Objects.hashCode(this.columnName);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ColumnHolder other = (ColumnHolder) obj;
        if (!Objects.equals(this.schemaName, other.schemaName)) {
            return false;
        }
        if (!Objects.equals(this.tableName, other.tableName)) {
            return false;
        }
        return Objects.equals(this.columnName, other.columnName);
    }

    @Override
    public String toString() {
        return String.format("%s.%s.%s", schemaName, tableName, columnName);
    }
}
