/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import ari.ucidy.UCD;
import ari.ucidy.UCDParser;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import javax.enterprise.context.Dependent;
import javax.inject.Inject;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Dependent
public class SearchUCDDialog implements Serializable {

    private static final long serialVersionUID = -3503024742241865133L;

    @Inject
    private ConfigurationManager config;

    @Inject
    private User user;

    private boolean manualInsertion;

    private String UCDManualText;

    private String description;
    private boolean UCDnotFound;
    private String UCDServiceErrorMessage;
    private String selectedUCD;
    private String suggestedUCD;
    private List<String> selectableUCDs;
    private List<UCDInfo> suggestedUCDs;

    private ParsedUCD parsedUCD;

    public SearchUCDDialog() {
        suggestedUCDs = new ArrayList<>();
        selectableUCDs = new ArrayList<>();
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public boolean isUCDnotFound() {
        return UCDnotFound;
    }

    public String getSelectedUCD() {
        return selectedUCD;
    }

    public String getSuggestedUCD() {
        return suggestedUCD;
    }

    public List<UCDInfo> getSuggestedUCDs() {
        return suggestedUCDs;
    }

    public void setDefault() {
        UCDManualText = null;
        parsedUCD = null;

        description = null;
        UCDnotFound = false;
        selectedUCD = null;
        suggestedUCD = null;
        suggestedUCDs.clear();
        selectableUCDs.clear();

        UCDServiceErrorMessage = null;
    }

    public void search() {
        try {
            String assignResponse = SearchUCD.assign(description);
            if (assignResponse == null) {
                UCDnotFound = true;
                selectedUCD = null;
                suggestedUCD = null;
                suggestedUCDs.clear();
            } else {
                selectedUCD = assignResponse;
                suggestedUCD = assignResponse;

                List<UCDInfo> suggestResponse = SearchUCD.suggest(description);
                if (suggestResponse == null) {
                    UCDnotFound = true;
                } else {
                    suggestedUCDs = suggestResponse;
                }
            }
        } catch (UCDServiceException e) {
            setUCDServiceErrorMessage(e);
        }
    }

    public void selectUCD(String selectedUCD) {
        this.selectedUCD = selectedUCD;
    }

    public boolean isManualInsertion() {
        return manualInsertion;
    }

    public void setManualInsertion(boolean manualInsertion) {
        this.manualInsertion = manualInsertion;
    }

    public String getUCDServiceErrorMessage() {
        return UCDServiceErrorMessage;
    }

    private void setUCDServiceErrorMessage(UCDServiceException e) {
        setDefault();
        UCDServiceErrorMessage = e.getMessage();
    }

    public String getUCDManualText() {
        return UCDManualText;
    }

    public void setUCDManualText(String UCDManualText) {
        this.UCDManualText = UCDManualText;
    }

    public void validateManualUCD() {
        if (UCDManualText == null || UCDManualText.isEmpty() || UCDManualText.trim().isEmpty()) {
            parsedUCD = null;
        } else {
            parsedUCD = new ParsedUCD(UCDManualText, config.getUCDConfiguration());
        }
    }

    public ParsedUCD getParsedUCD() {
        return parsedUCD;
    }

    public void rebuildUCD() {
        // Retrieving all words selected by the user.
        List<String> wordsToPermutate = new ArrayList<>();
        for (UCDInfo ucd : suggestedUCDs) {
            if (ucd.isSelected()) {
                wordsToPermutate.add(ucd.getWord());
            }
        }

        // Building UCD candidates list (all possible word permutations, including invalid UCDs)
        List<String> ucdCandidates = new ArrayList<>();
        for (int i = 1; i < 4; i++) {
            buildAllWordsPermutations(wordsToPermutate, ucdCandidates, i, 0, null);
        }

        // Filtering candidates list taking only fully valid UCDs
        selectableUCDs.clear();
        for (String ucdCandidate : ucdCandidates) {
            UCD ucd = UCDParser.parseUCD(ucdCandidate);
            if (ucd.isFullyValid()) {
                selectableUCDs.add(ucdCandidate);
            }
        }
    }

    /**
     * This recursive method allows to retrieve all UCD candidates for a given
     * list of UCD words. It computes all possible permutations of words that
     * could be an UCD of a given length. Resulting UCD candidates are added to
     * the result list passed as a parameters. These values are only candidates,
     * so they could be invalid UCD, so the list needs to be validated after its
     * creation. For the first recursive step set currentSize to 0 and
     * currentPerm to null.
     *
     * @param wordsList the list of words on which do the permutation
     * @param result the list in which store the result of the permutations
     * @param limit the number of words that will compose the UCD candidate
     * (permutation size limit)
     * @param currentSize number of words concatenated in the current recursive
     * step (it will stop the recursion when it will be equals to the limit
     * value)
     * @param currentPerm the word built by current recursive step
     */
    private void buildAllWordsPermutations(List<String> wordsList, List<String> result, int limit, int currentSize, String currentPerm) {
        if (currentSize == limit) {
            result.add(currentPerm);
        } else {
            for (String item : wordsList) {
                List<String> remaining = new ArrayList<>(wordsList);
                remaining.remove(item);
                String ucd;
                if (currentPerm != null) {
                    ucd = currentPerm + ";" + item;
                } else {
                    ucd = item;
                }
                buildAllWordsPermutations(remaining, result, limit, currentSize + 1, ucd);
            }
        }
    }

    public List<String> getSelectableUCDs() {
        return selectableUCDs;
    }
}
