/* 
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;

/**
 * Represents a Key entity belonging to a {@link TapSchema}.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class Key extends TapSchemaEntity implements Serializable {

    public static final String ID_KEY = "key_id";
    public static final String FROM_TABLE_KEY = "from_table";
    public static final String TARGET_TABLE_KEY = "target_table";

    private static final long serialVersionUID = -8783695875831579336L;

    private List<KeyColumn> keyColumns;
    private boolean visible;

    private String fromSchema;
    private String fromTable;
    private String targetSchema;
    private String targetTable;

    /**
     * Only for serialization.
     */
    private Key() {
        super();
    }

    /**
     * Default constructor.
     *
     * @see TapSchemaEntity
     */
    public Key(TapSchema tapSchema, Map<String, Object> metadata) {
        super(tapSchema, tapSchema.getTableModel(TapSchema.KEYS_TABLE), metadata);

        keyColumns = new ArrayList<>();
        visible = false;

        String[] fromSplit = ((String) metadata.get(FROM_TABLE_KEY)).split(Pattern.quote("."));
        String[] targetSplit = ((String) metadata.get(TARGET_TABLE_KEY)).split(Pattern.quote("."));

        fromSchema = fromSplit[0];
        fromTable = fromSplit[1];
        targetSchema = targetSplit[0];
        targetTable = targetSplit[1];
    }

    /**
     * Tells if a key has to be exposed by the TAP_SCHEMA. This happens when
     * both the key {@code from_column} and {@code target_column} have been
     * added to the TAP_SCHEMA.
     *
     * @return true if the column has to be exposed by the TAP_SCHEMA, false
     * otherwise.
     */
    public boolean isVisible() {
        return visible;
    }

    /**
     * Update the key visibility.
     *
     * @param visible true if the key has to be exposed by the TAP_SCHEMA, false
     * otherwise.
     */
    protected void setVisible(boolean visible) {
        this.visible = visible;
    }

    /**
     * Returns the identifier for this key. The TAP standard defines it as a
     * String, but TASMAN sets its value as an incremental integer (stored as a
     * String). Keys not already persisted have a null id.
     */
    public String getId() {
        return getValue(ID_KEY, String.class);
    }

    /**
     * Sets the identifier for this key.
     *
     * @see #getId
     */
    public void setId(String id) {
        setValue(ID_KEY, id);
        for (KeyColumn keyColumn : keyColumns) {
            keyColumn.setKeyId(id);
        }
    }

    /**
     * Returns all the {@code KeyColumn}s owned by this key.
     */
    public List<KeyColumn> getKeyColumns() {
        return Collections.unmodifiableList(keyColumns);
    }

    /**
     * Returns the name of the schema owning the foreign key.
     */
    public String getFromSchemaName() {
        return fromSchema;
    }

    /**
     * Returns the name of the table owning the foreign key (the name of the
     * table, without its schema name).
     */
    public String getFromTableSimpleName() {
        return fromTable;
    }

    /**
     * Returns the complete name of the table owning the foreign key (the schema
     * name plus the table name).
     */
    public String getFromTableCompleteName() {
        return getValue(FROM_TABLE_KEY, String.class);
    }

    /**
     * Returns the name of the schema owning the primary key.
     */
    public String getTargetSchemaName() {
        return targetSchema;
    }

    /**
     * Returns the name of the table owning the primary key (the name of the
     * table, without its schema name).
     */
    public String getTargetTableSimpleName() {
        return targetTable;
    }

    /**
     * Returns the complete name of the table owning the primary key (the schema
     * name plus the table name).
     */
    public String getTargetTableCompleteName() {
        return getValue(TARGET_TABLE_KEY, String.class);
    }

    /**
     * Adds a {@link KeyColumn} object to this key.
     *
     * @param fromColumn the name of the column owning the foreign key.
     * @param targetColumn the name of the column owning the primary key.
     */
    public KeyColumn addKeyColumn(String fromColumn, String targetColumn) {
        Map<String, Object> keyColumnMetadata = new HashMap<>();
        keyColumnMetadata.put(KeyColumn.FROM_COLUMN_KEY, fromColumn);
        keyColumnMetadata.put(KeyColumn.TARGET_COLUMN_KEY, targetColumn);
        KeyColumn keyColumn = new KeyColumn(tapSchema, this, keyColumnMetadata);
        keyColumns.add(keyColumn);
        return keyColumn;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void save() {
        if (!isVisible()) {
            initProperty(ID_KEY, null);
        }
        super.save();
        for (KeyColumn keyColumn : keyColumns) {
            keyColumn.save();
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();

        sb.append(String.format("[%s] %s(", getId(), getFromTableCompleteName()));

        boolean first = true;
        for (KeyColumn keyColumn : keyColumns) {
            if (!first) {
                sb.append(",");
            }
            first = false;
            sb.append(keyColumn.getFromColumn());
        }

        sb.append(String.format(") -> %s(", getTargetTableCompleteName()));

        first = true;
        for (KeyColumn keyColumn : keyColumns) {
            if (!first) {
                sb.append(",");
            }
            first = false;
            sb.append(keyColumn.getTargetColumn());
        }

        sb.append(")");

        return sb.toString();
    }

    @Override
    public int hashCode() {
        int hash = 7;
        for (KeyColumn keyColumn : keyColumns) {
            hash = 23 * hash + Objects.hashCode(keyColumn);
        }
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (Key.class != obj.getClass()) {
            return false;
        }

        final Key other = (Key) obj;
        List<KeyColumn> otherKeyColumns = other.getKeyColumns();

        // Comparing each key column
        if (keyColumns.size() != otherKeyColumns.size()) {
            return false;
        }
        for (int i = 0; i < keyColumns.size(); i++) {
            if (!keyColumns.get(i).equals(otherKeyColumns.get(i))) {
                return false;
            }
        }

        return true;
    }
}
