/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.webapp;

import it.inaf.ia2.tsm.Column;
import it.inaf.ia2.tsm.Status;
import it.inaf.ia2.tsm.Table;
import it.inaf.ia2.tsm.webapp.env.CustomPartialResponseWriter;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import javax.inject.Named;
import javax.json.Json;
import javax.json.JsonArrayBuilder;
import javax.json.JsonObjectBuilder;
import org.apache.deltaspike.core.api.scope.WindowScoped;

/**
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
@Named
@WindowScoped
public class ColumnsSorter implements Serializable {

    private static final long serialVersionUID = 1431422275331229274L;

    private static class ColumnHolder {

        String columnName;
        int columnIndex;

        ColumnHolder(String columnName, int columnIndex) {
            this.columnName = columnName;
            this.columnIndex = columnIndex;
        }
    }

    private List<ColumnHolder> sortedColumns;
    private List<String> unSortedColumns;

    public void openModal(Table table) {
        sortedColumns = new ArrayList<>();
        unSortedColumns = new ArrayList<>();

        for (Column column : table.getChildren(Status.ADDED_PERSISTED, Status.ADDED_NOT_PERSISTED)) {
            String columnName = column.getName();
            Integer columnIndex = (Integer) column.getProperty(Column.COLUMN_INDEX).getValue();
            if (columnIndex == null) {
                unSortedColumns.add(columnName);
            } else {
                sortedColumns.add(new ColumnHolder(columnName, columnIndex));
            }
        }

        Collections.sort(sortedColumns, new Comparator<ColumnHolder>() {
            @Override
            public int compare(ColumnHolder o1, ColumnHolder o2) {
                return Integer.compare(o1.columnIndex, o2.columnIndex);
            }
        });

        CustomPartialResponseWriter.getCurrentInstance().addCustomJSUpdate("main:columns_sorter", getJSONModel());
    }

    private JsonArrayBuilder getJABFromList(List<String> values) {
        JsonArrayBuilder jab = Json.createArrayBuilder();
        for (String value : values) {
            jab.add(value);
        }
        return jab;
    }

    private String getJSONModel() {
        JsonObjectBuilder job = Json.createObjectBuilder();
        job.add("sortedColumns", getJABFromList(getSortedColumns()));
        job.add("unSortedColumns", getJABFromList(unSortedColumns));
        return job.build().toString();
    }

    public List<String> getSortedColumns() {
        List<String> columns = new ArrayList<>();
        for (ColumnHolder ch : sortedColumns) {
            columns.add(ch.columnName);
        }
        return columns;
    }

    public List<String> getUnSortedColumns() {
        return unSortedColumns;
    }
}
