/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm.model;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Utility class for retrieving models of schemata managed by TASMAN.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class SchemaModels {

    private static final String SCHEMA_DEFINITION_FOLDER = "schema_definition";

    // Map keys are version numbers
    private static final Map<String, SchemaModel> TAP_SCHEMA_MODELS;
    private static final Map<String, SchemaModel> IVOA_SCHEMA_MODELS;

    private SchemaModels() {
    }

    static {
        try {
            Properties properties = getCoreProperties();

            String[] tapSchemaModelFiles = getXMLModelFileNames(properties, "tap_schema_models");
            String[] ivoaModelFiles = getXMLModelFileNames(properties, "ivoa_schema_models");

            TAP_SCHEMA_MODELS = new XMLModelsLoader(tapSchemaModelFiles).load();
            IVOA_SCHEMA_MODELS = new XMLModelsLoader(ivoaModelFiles).load();
        } catch (IOException e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    private static Properties getCoreProperties() throws IOException {
        Properties props = new Properties();
        try (InputStream in = SchemaModels.class.getClassLoader().getResourceAsStream("core.properties")) {
            props.load(in);
        }
        return props;
    }

    private static String[] getXMLModelFileNames(Properties props, String propertyKey) throws IOException {
        String[] fileNames = props.getProperty(propertyKey).split(",");

        String[] modelFiles = new String[fileNames.length];
        for (int i = 0; i < fileNames.length; i++) {
            modelFiles[i] = String.format("%s%s%s", SCHEMA_DEFINITION_FOLDER, File.separator, fileNames[i].trim());
        }

        return modelFiles;
    }

    public static SchemaModel getTapSchemaModel(String version) {
        return TAP_SCHEMA_MODELS.get(version);
    }

    public static Iterable<SchemaModel> getTapSchemaModels() {
        return TAP_SCHEMA_MODELS.values();
    }

    /**
     * Returns the {@link TableModel} of TAP_SCHEMA table.
     *
     * @param tableName the name of the table.
     * @param version TAP_SCHEMA version.
     */
    public static TableModel getTapSchemaTableModel(String tableName, String version) {
        return getTapSchemaModel(version).getTable(tableName);
    }

    private static List<String> getAvailableVersions(Map<String, SchemaModel> schemaModels) {
        List<String> versions = new ArrayList<>();
        for (SchemaModel tapSchemaModel : schemaModels.values()) {
            versions.add(tapSchemaModel.getVersion());
        }
        return versions;
    }

    public static List<String> getAvailableTapSchemaVersions() {
        return getAvailableVersions(TAP_SCHEMA_MODELS);
    }

    public static List<String> getAvailableIvoaSchemaVersions() {
        return getAvailableVersions(IVOA_SCHEMA_MODELS);
    }

    public static SchemaModel getIvoaSchemaModel(String version) {
        return IVOA_SCHEMA_MODELS.get(version);
    }

    public static Iterable<SchemaModel> getIvoaSchemaModels() {
        return IVOA_SCHEMA_MODELS.values();
    }
}
