/*
 * _____________________________________________________________________________
 * 
 * INAF - OATS National Institute for Astrophysics - Astronomical Observatory of
 * Trieste INAF - IA2 Italian Center for Astronomical Archives
 * _____________________________________________________________________________
 * 
 * Copyright (C) 2017 Istituto Nazionale di Astrofisica
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package it.inaf.ia2.tsm;

import java.io.Serializable;

/**
 * Models an inconsistency in a column definition detected in an existing
 * column. This happens when the column datatype read from the database metadata
 * is different from which is expected according the {@link ColumnModel}
 * defining that column. To fix this issue an {@code ALTER TABLE} is necessary.
 *
 * @author Sonia Zorba {@literal <zorba at oats.inaf.it>}
 */
public class WrongDataType implements Serializable {

    private static final long serialVersionUID = 1541867434766884291L;

    private final ColumnHolder columnHolder;
    private final String wrongDataType;
    private final String correctDataType;
    private final String adqlCorrectDataType;
    private final Integer size;

    /**
     * Default constructor.
     *
     * @param columnHolder a representation for the column having a wrong
     * datatype.
     * @param wrongDataType the current (wrong) column datatype.
     * @param correctDataType the expected column datatype.
     * @param adqlCorrectDataType the expected column datatype, using ADQL
     * datatype syntax.
     * @param size the expected column size.
     */
    public WrongDataType(ColumnHolder columnHolder, String wrongDataType, String correctDataType, String adqlCorrectDataType, Integer size) {
        this.columnHolder = columnHolder;
        this.wrongDataType = wrongDataType;
        this.correctDataType = correctDataType;
        this.adqlCorrectDataType = adqlCorrectDataType;
        this.size = size;
    }

    /**
     * Returns the representation of the column having the wrong datatype.
     */
    public ColumnHolder getColumnHolder() {
        return columnHolder;
    }

    /**
     * Returns the current (wrong) datatype, as retrieved from the database
     * metadata.
     */
    public String getWrongDataType() {
        return wrongDataType;
    }

    /**
     * Returns the expected datatype, according to the {@link ColumnModel}
     * associated to this column.
     */
    public String getCorrectDataType() {
        return correctDataType;
    }

    /**
     * Returns the expected datatype, using ADQL datatype syntax.
     */
    public String getAdqlCorrectDataType() {
        return adqlCorrectDataType;
    }

    /**
     * Returns the desired columns size.
     */
    public Integer getSize() {
        return size;
    }
}
