__DESCRIPTION__="""

TAP services for ARTECS

By M.Maris 1.1 - 2019 Nov 15 - 
Ported to python3
Needs pyvo 



Example:
   >>> import artecs
   >>> atap=artecs.exop_pubblic_tap()
   
   >>> atap.EXPLAIN()
   
   >>> atap.keys()


   >>> tab=atap.search('(0.7 <= SMA) and (SMA <=3.)')

   >>> tab.FO_CONST.unique()

   >>> tab.to_csv('/tmp/pippo.csv',sep=' ')

   >>> MAP=atap.get_map(tab.URL[0])

"""

try :
   import pyvo
   BAD=False
except :
   print("PYVO not found, to install pyvo : \n\t>sudo pip install pyvo\n%s EXOP_TAP class will not work properly"%("/".join(__file__.split('/')[-2:])))
   BAD=True

#if BAD : 
   #import sys
   #sys.exit(1)
#del BAD

class EXOP_TAP :
   def __init__(self,tap_url="http://archives.ia2.inaf.it/vo/tap/exo",table_name="exo.EXO",temporary_files_path='/tmp') :
      #import pyvo
      #
      #creates empty instantiation
      self._empty()
      #
      self._temporary_files_path=temporary_files_path+'/' if temporary_files_path!='' and temporary_files_path!=None else ''
      #
      #connects to db 
      self._url=tap_url
      self._table_name=table_name
      #
      #template for query_string, two fields: <QUERY> <FIELDS>
      self._template_query="SELECT %s %s FROM "+self._table_name
      #
      try :
         self._tap_service = pyvo.dal.TAPService(tap_url)
      except :
         print ("Error, unable to connect to TAP_URL '"+tap_url+"'")
      #
      #from db gets fields description
      self._get_field_description()
   #
   def _empty(self) :
      self._url=None
      self._table_name=None
      self._tap_service = None
      self._elapsed_time=0.
      self._search_result=None
      self._field_description=None
      self._field_names=None
      self._search_success=None
      self._template_query=None
      self._download_success=None
   #
   def _get_field_description(self) :
      """ gets description of fields """
      self.adql_search('SELECT TOP 1 * FROM '+self._table_name)
      self._field_description=self._search_result.fielddescs
      self._field_names=self._search_result.fieldnames
      self.clean()
   #
   def keys(self) :
      """list of fields in the database"""
      return self._field_names
   #
   def has_key(self,this) :
      """has_key"""
      return this in self._field_names
   #
   def __include__(self,this) :
      """this is an object"""
      return this in self._field_names
   #
   def success(self) : 
      """returns True if last search was successfull"""
      return self._search_success==True
   #
   def clean(self) :
      """clean information from last search"""
      self._search_success=None
      self._search_result=None
   #
   def adql_search(self,adql_string) :
      """search on database using ADQL string """
      import time
      self.clean()
      self._search_success=False
      self._elapsed_time=time.time()
      self._search_result=self._tap_service.search(adql_string)
      self._elapsed_time=time.time()-self._elapsed_time
      self._search_success=True
   #
   def search_string(self,SELECTION,TOP,FIELDS,WHERE,SORT) :
      """creates a properly formatted adql query_string"""
      adql='SELECT' 
      adql+='' if type(SELECTION)==type(None) else ' '+SELECT
      adql+=' TOP %d'%TOP if type(TOP)==type(1) else '' 
      adql+=' *' if type(FIELDS)==type(None) else ' '+FIELDS
      adql+=' FROM '+self._table_name 
      adql+='' if WHERE=='' or WHERE==None else ' WHERE '+WHERE
      adql+='' if SORT=='' or SORT==None else ' SORT BY '+SORT
      return adql
   #
   def search(self,WHERE,SELECTION=None,FIELDS=None,TOP=None,SORT=None,as_astropy=False,as_votable=False) :
      """download a table from the database
         search('') or search(WHERE='') returns all the data in the database
         
         the table can be returned as :
            pandas dataframe (default)
            astropy table (as_astropy = True)
            votable (as_votable=True)
      """
      import time
      #
      adql=self.search_string(SELECTION,TOP,FIELDS,WHERE,SORT)
      self.adql_search(adql)
      #
      if as_astropy :
         return self._search_result.table
      elif as_votable :
         return self._search_result.votable
      else :
         return self._search_result.table.to_pandas()
   #
   def get_map(self,_URL,outfile=None) :
      """gets a map from an URL"""
      import numpy as np
      import time
      from .artecs_map import artecs_map
      try :
         from urllib2 import urlopen
      except :
         from urllib.request import urlopen
      import requests
      import gzip
      import os
      try :
         from StringIO import StringIO
      except :
         from io import StringIO
      if type(_URL) == type("") :
         URL=_URL+''
      else :
         URL=_URL.decode("utf-8")
      #
      if type(outfile) != type('') :
         #if out file not specified creates a unique temporary name and reserves it
         flag = True
         while flag :
            tmp=URL+' '+time.asctime()
            tmp=tmp+' '+str(np.random.randint(1000000))
            tmp=abs(hash(tmp))
            tmp=hex( tmp )[2:]
            _o=self._temporary_files_path+'artecs_download_file_'+tmp+'.fits'
            flag=os.path.isfile(_o)
            open(_o,'w').write('a') #this is to reserve the file name
      else :
         _o=outfile
      #
      tic=time.time()
      request=requests.Request(URL)
      #response = urlopen(request)
      try :
         response = urlopen(URL)
      except :
         print("URL %s not found"%URL)
         if type(outfile) != type('') :
            os.remove(_o)
         self._download_success=False
         return
      #buf = StringIO(response.read())
      buf = response
      f = gzip.GzipFile(fileobj=buf)
      data=f.read()
      open(_o,'wb').write(data)
      tic=time.time()-tic
      #
      amap=artecs_map(_o)
      amap.url=URL
      if type(outfile) != type('') :
         os.remove(_o)
      self._download_success=True
      return amap
   #
   def download_map(self,URL,outfile=None,path=None) :
      """download the fits file corresponding at a given URL
      if outfile is not specified the fits file name from the URL is used
      if path is not specified the current file is stored in the current path
      """
      _path='' if type(path) != type('') else path+'/'
      if type(outfile)!=type('') :
         _o=URL.split('/')[-1].split('.gz')[0]
         _o=_path+_o
      else :
         _o=outfile
      self.get_map(URL,outfile=_o)
   # 
   def EXPLAIN(self) :
      """print a short introduction on the ADQL query language"""
      print ("""
ADQL = Astronomical Data Query Language

A query is a string 
- SELECTION set of data to be selected
- HOW_MANY elements to select
- WHICH_FIELDS fields to select
- FROM which table
- WHERE condition for selection is true

Example:

1. Download all the data
   SELECT * FROM exo.EXO
   
2. Download only the first 10 elements
   SELECT TOP 10 * FROM exo.EXO

3. Download only the first 10 elements with SMA in the range 0.9 to 1.1
   SELECT TOP 10 * FROM exo.EXO WHERE SMA BETWEEN 0.9 AND 1.1
   
   alternativelly

   SELECT TOP 10 * FROM exo.EXO WHERE (0.9 <= SMA) AND (SMA <= 1.1)

4. Download only the first 10 elements with SMA in the range 0.9 to 1.1 and CONTHAB>=0.5
   SELECT TOP 10 * FROM exo.EXO WHERE SMA BETWEEN 0.9 AND 1.1 AND CONTHAB>=0.5

5. Arithmetic calculations are allowed if needed for selection so to download the first 10 elements with SMA^1.5 > 0.87 
   SELECT TOP 10 * FROM exo.EXO WHERE power(SMA,1.5)> 0.87 
   
6. returns just columns SMA and CONTHAB from previous example:
   SELECT TOP 10 SMA,CONTHAB FROM exo.EXO WHERE power(SMA,1.5)> 0.87 
   
Note that the query is not sensitive to uppercase or lowercase.

Tutorials: 
   http://www.g-vo.org/tutorials/gaia-mock-tap.pdf
   http://www.ivoa.net/documents/REC/ADQL/ADQL-20081030.pdf
   http://www.ivoa.net/documents/

""")

