;+
; :Author: schreibl
;-
; $Id: zernike_cube.pro#1.1 $
;
;+
; NAME:
; ZERNIKE_CUBE
;
; PURPOSE:
; This function computes a cube of Zernike polynomials (piston excluded).
; The polynomials are defined over a square or circular domain.
; They are normalized to unit RMS over the circular domain.
; The circular domain could have a central obstruction: the polynomials 
; are masked accordingly (however they are not the anular Zernike polymials).
;
; CATEGORY:
; Mathematics.
;
; CALLING SEQUENCE:
; Result = ZERNIKE_CUBE(Npix, Diameter, Nzer)
;
; INPUTS:
; Npix: size of square domain.
; 
; Diameter: diameter of pupil mask.
;
; Nzer: number of Zernike polynomials (excluding piston).
; 
; KEYWORD PARAMETERS:
; OBSTRUCTION: relative central obstruction of pupil mask.
; 
; MASK: set this binary keyword to multiply the polynomials by the pupil mask.
; 
; OUTPUTS:
; The function returns a double precision floating point 3D array 
; of size Npix*Npix*Nzer.
;
; OPTIONAL OUTPUTS:
; M: pupil mask used for the calculation of the Zernike polynomials.
; 
; X, Y: 2D arrays of coordinates used for the calculation of the 
;    Zernike polynomials.
; 
; PROCEDURE:
; Based on routine ZERNIKE.PRO.
;
; EXAMPLE:
;
; MODIFICATION HISTORY:
;   Written by: E.D. (INAF-OABo), April 2011.
;   1) E.D., May 2011: fixed bug on normalized coordinates.
;-

function zernike_cube, npix, diameter, OBSTRUCTION = o, nzer, $
                       MASK = mask, m, x, y


  m = pupilmask(npix, diameter, OBSTRUCTION = o, x, y)   
 
if keyword_set(mask) then begin
  m = mask
  diameter = (size(m,/dim))[0]
endif
w = where(m)
zer = dblarr(npix, npix, nzer)
x1 = x / (diameter/2.0)
y1 = y / (diameter/2.0)
for n = 0L, nzer-1 do begin
   zer[*,*,n] = zernike(n + 2, x1, y1)
   zer[*,*,n] = zer[*,*,n] - mean((zer[*,*,n])[w])
   zer[*,*,n] = zer[*,*,n] / sqrt(total((zer[*,*,n])[w]*(zer[*,*,n])[w])/total(m));stdev((zer[*,*,n])[w])
endfor

;if keyword_set(mask) then $
;   for n = 0L, nzer-1 do zer[*,*,n] = zer[*,*,n] * m

return, zer
end
