;+
; NAME:
;   FINDNM
;
; PURPOSE:
; Find the radial and azimuthal order of one zernike's polynomial of degree j
;
; CATEGORY:
;
;
; CALLING SEQUENCE:
; out = findnm(var1)
;
; INPUTS:
; var1    : polynomial degree
;
; OUTPUTS:
; out1    : array with the radial and azimuthal orders
;
; SIDE EFFECTS:
;
;
; MODIFICATION HISTORY:
;   Written by:  Massimiliano Tordi, Jan 2000.
;-
;
Function FindNM,j

n = 0
Repeat Begin
        j_max = (n+1)*(n+2)/2
        n = n+1
EndRep Until j_max GE j
n = n-1
;
;  Finds the order for which m = 0, given n even
;  and establishes if n is even or odd.
;  Uses this property:
;  mode (m = 0)     1   4   11    22     37
;  difference         3   7    11    ...
;  second diRfference    4    4    ...
;
j_0 = (n*n+n+2)/2
Parity = n - 2*Fix(n/2)
;
;  Finds m
;
If ((j EQ j_0) And (Parity EQ 0)) Then Begin
 m = 0
EndIf Else Begin
 l = j_max
 k = 0
 While (l GT j) Do Begin
  l = l - 1
  k = k + 1
 EndWhile
 m = n - 2*Fix(k/2.)
EndElse

Coeff = [n,m]

Return,Coeff

End
; $Id: zernike.pro, v 1.0 Jan 2000 m.t. $
;
;+
; NAME:
;   RHO
;
; PURPOSE:
; Radial coordinate in the pupil plane
;
; CATEGORY:
;
;
; CALLING SEQUENCE:
; out = rho(arr1,arr2)
;
; INPUTS:
; arr1    : array with the x-coordinates in the pupil plane
; arr2    : array with the y-coordinates in the pupil plane
;
; OUTPUTS:
; out1    : array with the radial coordinate of the points (x,y)
;
; SIDE EFFECTS:
;
;
; MODIFICATION HISTORY:
;   Written by:  Massimiliano Tordi, Jan 2000.
;-
;
Function Rho, X,Y

Return, Sqrt(Double(X)^2+Double(Y)^2)
End
;+
; NAME:
;   THETA
;
; PURPOSE:
; Angular coordinate in the pupil plane
;
; CATEGORY:
;
;
; CALLING SEQUENCE:
; out = rho(arr1,arr2)
;
; INPUTS:
; arr1    : array with the x-coordinates in the pupil plane
; arr2    : array with the y-coordinates in the pupil plane
;
; OUTPUTS:
; out     : array with the angular coordinate of the points (x,y)
;
; SIDE EFFECTS:
;
;
; MODIFICATION HISTORY:
;   Written by:  Massimiliano Tordi, Jan 2000.
;-
;
Function Theta, X,Y

Return, Atan(-Y,-X) + !DPI
End

;+
; NAME:
;   RADIAL
;
; PURPOSE:
; Find the radial part of a Zernike's polynomial of degree j
;
; CATEGORY:
;
;
; CALLING SEQUENCE:
; out = radial(var1,var2,array1)
;
; INPUTS:
; var1    : polynomial radial order
; var2    : polynomial azimuthal order
; array1  : points of the pupil plane where the polynomial
;           has to be evaluated
;
; OUTPUTS:
; out1    : array with the radial part evaluated in the given points
;
; SIDE EFFECTS:
;
;
; MODIFICATION HISTORY:
;   Written by:  Massimiliano Tordi, Jan 2000.
;-
;
Function Radial,n,m,R

Sup = (n-m)/2
Sum = 0.d0

For I = 0,Sup Do Begin
 Sum = Sum + (-1)^I*Factorial(n-I)/Factorial(I)/$
 Factorial((n+m)/2-I)/Factorial((n-m)/2-I)*R^(n-2.*I)
EndFor

Return,Sum
End
;+
; NAME:
;   ZERNIKE
;
; PURPOSE:
; Calculate the zernike's polynomial of degree j in the points (x,y)
;
; CATEGORY:
;
;
; CALLING SEQUENCE:
; out = zernike(var1,arr1,arr2)
;
; INPUTS:
; var1    : polynomial degree
; arr1    : array with the x (cartesian) coordinates of the points
;           where the polynomial should be evaluated
; arr2    : array with the y (cartesian) coordinates of the points
;           where the polynomial should be evaluated
;
; OUTPUTS:
; out1    : array with the radial and azimuthal orders
;
; SIDE EFFECTS:
;
;
; MODIFICATION HISTORY:
;   Written by:  Massimiliano Tordi, Jan 2000.
;-
;
Function Zernike,j,x,y

Coeff = FindNM(j)
n = Coeff[0]
m = Coeff[1]
;
; finds the radial and angular coordinate
;
t = Theta(X,Y)
r = Rho(X,Y)
;
;  Finds the polynomial: Radial gives the radial part
;
If (m EQ 0) Then Znk = Sqrt(n+1.D0)*Radial(n,0,r)
If (m GT 0) Then Begin
Parity = j-2*Fix(j/2)
Case Parity Of
 1: Znk = Sqrt(n+1.D0)*Sqrt(2.D0)*Radial(n,m,r)*Sin(m*t)
 0: Znk = Sqrt(n+1.D0)*Sqrt(2.D0)*Radial(n,m,r)*Cos(m*t)
EndCase
EndIf
;
;  Normalization
;
;   A = Where(Pupil EQ 1)
;   Return,Znk/(Max(Znk(A))-Min(Znk(A)))
   ;Return,Znk/Max([Max(Znk(A)),-Min(Znk(A))])
Return,Znk
End