;+ $Id: slope_to_wavefront_ft.pro, v 0.5 May 2011 $
;
; ROUTINE NAME:
; slope_to_wavefront_ft
;
; PURPOSE:
; Convert wavefront slopes to wavefront map by an iterative Fourier method.
;
; CALLING SEQUENCE:
; result = slope_to_wavefront_ft(sx_in, sy_in, mask_in)
;
; INPUTS:
; sx_in, sy_in: square arrays of slopes (X and Y wavefront derivatives)
; mask_in: square array (byte or integer or float) with mask to define support of sx and sy
;    (1: inside support; 0 outside support)
; 
; OPTIONAL INPUTS:
; NO_EXTEND: By default the arrays are extended by 0-padding to improve 
;    reconstruction performance. Set /NO_EXTEND to avoid extension.
; MAXITER: maximum number of iterations (default = 100)
;
; OUTPUTS:
; Return reconstructed wavefront map (same size as sx_in, sy_in, mask_in).
; The reconstructed wavefront has zero mean over the pixels defined by mask_in.
; 
; OPTIONAL OUTPUTS:
; ITER: set this keyword to a named variable to retrieve number of iterations.
; ERR: set this keyword to a named variable to retrieve RMS error of last to 
;      second-last wavefront estimate
; 
; METHOD:
; Adapted from F. Roddier & C. Roddier, 1991, Applied Optics 30, 1325.
;
; RESTRICTIONS:
; Arrays are assumed to be square.
; 
; EXAMPLES:
; Given arrays of slopes Sx and Sy and mask M, reconstruct wavefront W:
; IDL> W = SLOPE_TO_WAVEFRONT_FT(Sx*M, Sy*M, M)
; 
; MODIFICATION HISTORY
; Written by: E.D. (INAF-OABO), 200x, as SIGNAL_TO_WAVEFRONT.
; 1) E.D. (INAF-OABo), April 2011: revised and changed to slope_to_wavefront
; 2) E.D. (INAF-OABO), April 2011: size extension set as default; introduced
;    keyword NO_EXTEND.
; 3) E.D. (INAF-OABo), April 2011: renamed SLOPE_TO_WAVEFRONT_FT
; 4) E.D. (INAF-OABo), May 2011: output same type as input slopes
;-

function slope_to_wavefront_ft, sx_in, sy_in, mask_in, $
            NO_EXTEND = no_extend, MAXITER = maxiter, $
            ITER = iter, ERR = err

on_error, 2

if n_elements(maxiter) eq 0 then maxiter = 100
;if n_elements(tol) eq 0 then tol = 1e-5

sx = sx_in
sy = sy_in
mask = mask_in
s = (size(mask, /DIM))[0]
if not keyword_set(no_extend) then begin
   s = s * 2
   sx = extend_array(sx, s, s)
   sy = extend_array(sy, s, s)
   mask = extend_array(mask, s, s)
endif

w = where(mask and 1B)
wout = where((1 - mask) and 1B)
converg = 0B
iter = 0L
while iter lt maxiter and not converg do begin
   iter = iter + 1
   l = deriv(sx) + rotate(deriv(rotate(sy,3)),1)
   u = rebin(frequency(s), s, s)
   v = rebin(transpose(frequency(s)), s, s)
   parab = -(2 * !pi)^2 / s^2 * (u^2 + v^2)
   parab[0,0] = 1
   l = fft(l) / parab
   wf = real_part(fft(l, /INVERSE))
   wf[w] = wf[w] - mean(wf[w])
   sx[wout] = (deriv(wf))[wout]
   sy[wout] = (rotate(deriv(rotate(wf,3)),1))[wout]
;   if iter gt 1 then converg = stdev((wf - wf0)[w]) / stdev(wf0[w]) lt tol
   if iter gt 1 then err = stdev((wf - wf0)[w]) / stdev(wf0[w])
   if iter gt 2 then converg = err ge err0
   if not converg then begin
      wf0 = wf
      if iter gt 1 then err0 = err
   endif else begin
      wf = wf0
      err = err0
   endelse
;   if iter gt 1 then print, err, converg, iter
 endwhile

wf = wf * mask

s = (size(mask_in, /DIM))[0]
if not keyword_set(no_extend) then $
   wf = sub_array(wf, s, s, REFERENCE = [s, s])

return, wf
end
