; $Id: modes.pro#0.2 $
;
;+
; NAME:
; MODES
;
; PURPOSE:
; Compute the X and Y derivatives of a set of basis functions defined on 
; a circular domain, chosen among Zernike or Karhunen-Loeve polynomials.
;
; CATEGORY:
; Modal analysis.
;
; CALLING SEQUENCE:
; MODE_DERIV, Mode_type, Mode_num, Npix, Diameter, Nsamp, $
;    Mask, Z, Der_x, Der_y
;
; INPUTS:
; Mode_type: String parameter to choose function type:
;    "Z": Zernike polynomials
;    "KL": Karhunen-Loeve polynomials.
; 
; Mode_num: Number of modes to compute and derive. In case Zernike 
;    polynomials are selected, the piston mode is excluded.
; 
; Npix: Size of square array containing a given mode and its 
;    X and Y derivatives.
; 
; Diameter: Diameter of circular domain of basis functions (pixels).
; 
; KEYWORD PARAMETERS:
; OBSTRUCTION: Relative central obstruction of circular domain.
;    Default OBSTRUCTION = 0.0.
;
; NSAMP: Oversampling factor for calculations. Default NSAMP = 1.
; 
; NOTILT: Set this binary keyword to exclude tip and tilt modes. Only
;    valid for Zernike polynomials.
;
; NRAD: Number of radial points for calculation of K-L polynomials. 
;    Default NRAD = 100.
;
; NAZIM: Number of azimuthal points for calculation of K-L polynomials. 
;    Default NAZIM = 500.
;    
; DER: Set this binary keyword to calculate the derivatives of the modes.
;
; OUTPUTS:
; Mask: Floating-point array defining circular domain (size Npix*Npix).
;    If NSAMP = 1, the non-zero pixels of Mask all have value = 1.
;    If NSAMP > 1, the non-zero pixels of Mask could have a value < 1, 
;    as a result of the REBIN function (see PROCEDURE below).
; 
; Z: Cube (size Npix*Npix*Mode_num) of basis functions. Type is double-
;    precision for Zernike polynomials, single-precision for K-L.
;
; Der_x, Der_y: Double-precision arrays of derivatives 
;    (size Npix*Npix*Mode_num). Available only if keyword /DER is set.
; 
; RESTRICTIONS:
; Zernike polynomials are calculated by routine ZERNIKE_CUBE.
; K-L polynomials are calculated by routine K-L.
; All restrictions reported therein apply here.
; Polyomials are exactly normalized only if NSAMP = 1. If NSAMP > 1 the 
; normalization is not perfect.
;
; PROCEDURE:
; The polynomials are defined over a circular domain slightly larger 
; than required to avoid edge effect in the derivatives. 
; X and Y derivatives are computed by IDL DERIV. 
; Results are cut to match desired size. 
; If an oversampling factor is used, the results are binned to the 
; desired size by IDL REBIN.
;
; EXAMPLE:
; Compute the derivatives of the first 10 K-L polynomials defined over 
; a circular domain of 16 pixel diameter on a 16*16 matrix without 
; oversampling factor:
; IDL> modes,"KL",10,16,16,m,z,dx,dy
; 
; Same as above, but for Zernike polynomials:
; IDL> modes,"Z",10,16,16,m,z,dx,dy
; 
; MODIFICATION HISTORY:
;   Written by: L.S. (UniBo), 2010, as Z_DERIVATIVE.
;   1) E.D. (INAF-OABo), April 2011. Changed to MODE_DERIV, including K-L.
;   2) E.D. (INAF-OABo), April 2011. Changed to MODES: calculation of 
;      derivatives is optional.
;-

pro modes, mode_type, mode_num, npix, diameter, OBSTRUCTION = o, $
           NSAMP = nsamp, NOTILT = notilt, NRAD = nrad, NAZIM = nazim, $
           DER = der, MASK = mask, m, z, der_x, der_y

;on_error, 2

if n_elements(o) eq 0 then o = 0.0
if n_elements(nsamp) eq 0 then nsamp = 1
if n_elements(nrad) eq 0 then nrad = 100
if n_elements(nazim) eq 0 then nazim = 500
if keyword_set(mask) then mask_ex = extend_array(mask,(size(mask,/dim))[0]+2,(size(mask,/dim))[1]+2)

npix_s = npix * nsamp
if nsamp gt 1 then begin
  if keyword_set(mask) then mask_ex = rebin(mask_ex,nsamp*((size(mask_ex,/dim))[0]+2), nsamp*((size(mask_ex,/dim))[0]+2))
endif
diameter_s = diameter * nsamp
znum = mode_num
case strupcase(mode_type) of
   "Z": begin
      z = zernike_cube(npix_s + 2, diameter_s, OBSTRUCTION = o, znum, mask = mask_ex, m)
      if keyword_set(notilt) then begin
         z = z[*,*,2:*]
         znum = znum - 2
      endif
   end
   "KL": begin
      z = KL(mode_num, npix_s + 2, diameter_s/2.0, o, m, NRAD = nrad, NAZIM = nazim)
   end
   else: begin
      message, "Unknown basis mode type"
      return
   end
endcase


m = rebin(float(m[1:npix_s,1:npix_s]), npix, npix) 
zcube = make_array(npix, npix, znum, /DOUBLE)
if keyword_set(der) then begin
   der_x = make_array(npix, npix, znum, /DOUBLE)
   der_y = make_array(npix, npix, znum, /DOUBLE)
   dx = make_array(npix_s, npix_s, /DOUBLE)
   dy = make_array(npix_s, npix_s, /DOUBLE)
   for i = 0L, znum-1 do begin
      for j = 1L, npix_s do begin
         dx[*,j-1] = (deriv(z[*,j,i]))[1:npix_s]
         dy[j-1,*] = (deriv(z[j,*,i]))[1:npix_s]
      endfor
      der_x[*,*,i] = nsamp * (rebin(dx, npix, npix) * m)
      der_y[*,*,i] = nsamp * (rebin(dy, npix, npix) * m)
   endfor
endif
z = rebin(z[1:npix_s,1:npix_s,*], npix, npix, znum) ;* $
   ; (rebin(mask, npix, npix, znum) and 1B)

return
end
