;+
; NAME:
;	KL
;
; PURPOSE:
; Compute Karhunen-Loeve polynomials defined over a circular domain.
;
; CALLING SEQUENCE:
;	KLcube = KL(npoly, siz, radius, obsratio, pupil_mask, $
;               NRAD = nrad, NAZIM = nazim, STF = stf)
;
; INPUTS:
;   npoly: number of K-L polynomials to generate
;
;   siz: size of square array where each polynomial is defined
;
;   radius: radius of pupil (circular domain) in pixels
;
;   obsratio: fractional central obstruction (e.g. 0.3)
;
; KEYWORD PARAMETERS:
;   NRAD: number of radial points to compute K-L in polar coordinates
;
;   NAZIM: number of azimuthal points to compute K-L in polar coordinates
;
;   STF: string, representing name of function to define a turbulence
;        spectrum. The default is "kolstf" (Kolmogorov spectrum)
;
; OUTPUTS:
;   KLcube: cube of K-L polynomials of dimension siz*siz*npoly.
;      The polynomials have mean=0 and rms=1 over the circular domain.
;
; OPTIONAL OUTPUTS:
;   pupil_mask: binary array with pupil
;
; PROCEDURE:
;   Define K-L polynomials in polar coordinates then convert to
;   cartesian coordinates by interpolation. Interpolated polynomials
;   are re-normalized to STDEV = 1.
;-



function KL, npoly, siz, radius, obsratio, pupil_mask, $
             NRAD = nrad, NAZIM = nazim, STF = stf

; Define default radial and azimuthal sampling
if not keyword_set(nrad) then nrad = long(sqrt(8 * npoly))
if not keyword_set(nazim) then nazim = long(5 * nrad)
;print, FORMAT = "(A, I4, A, I4, A)", $
;       "Computing K-L basis with ", nrad, "   radial and ", nazim, "   azimuthal points"
;print, ""

; Compute K-L basis and define polar-to-cartesian coordinate transformation
basis = gkl_bas(RI = obsratio, NR = nrad, NP = nazim, NFUNC = npoly, STF = stf)
margin = float(siz - 2 * radius) / 2
pctrans = set_pctr(basis, NCP = siz, NCMAR = margin)
pupil_mask = pctrans.ap
pupil_index = where(pupil_mask)

; Map K-L from polar to cartesian coordinates and re-normalize to RMS = 1
KLcube = fltarr(siz, siz, npoly)
for n = 0L, npoly - 1 do begin
   KLn = pol2car(pctrans, gkl_sfi(basis, n), /MASK)
   for it = 1, 2 do KLn = KLn - mean(KLn[pupil_index])
   for it = 1, 2 do KLn = KLn / sqrt(total(KLn[pupil_index]*KLn[pupil_index])/total(pupil_mask)) ;stdev(KLn[pupil_index])
   KLcube[*,*,n] = Kln
endfor

return, KLcube
end
