;+
;    FUNCTION gkl_bas, ri=ri, nr=nr, np=np, nfunc=nfunc, stf=stf
;
;    Generalised Karhunen-Loeve functions
;
;    generate the furst nfunc Karhunen-Loeve functions for an annular
;    apperture spanning radii ri to 1, for the structure function
;    'stf'
;    The results are in polar coordinates with nr radial points, and
;    np azimuthal ones.
;
;    returns gklbasis:
;
;    gklbasis = {$
;                 nr:nr, np:np, nfunc:nfunc, $
;                 ri:ri, stfn:' ',  $
;                 radp:radp, $
;                 evals:evals, nord:nord, npo:npo, ord:ord,  $
;                 rabas:rabas, azbas:azbas $
;                 }
;
;    KEYWORDS
;         (all optional, see the code for the default values)
;         ri  -  the radius of the central obscuration
;         nr  -  number of radial points for the basis elements
;         np  -  number of azimuthal points
;         nfunc - number of functions to be generated
;         stf - an idl function giving the structure function in terms
;               of (r/D) ie, with arguments between 0 and 1
;
;-







FUNCTION gkl_radii, nr, ri
    d = (1. - ri^2) / nr
    rad = ri^2 + d/2 + d * findgen (nr)
    rad = sqrt (rad)
    return, rad
END


FUNCTION gkl_mkker, ri, nr, rad,  stf
                               ;  the  kernel constructed here should be
                               ; simply a discretization of the
                               ; continuous kernel.  It  needs rescaling
                               ; before  it  is trated  as a matrix for
                               ; finding  the eigen-values

    nth = 5   * nr
    kers  = fltarr  (nr, nr, nth)
    cth = cos  (findgen (nth) * (2.  * !pi / nth))
    dth = 2.  * !pi / nth
    fnorm = -1. / (2 * !pi * (1. - ri^2)) * 0.5
                               ; the 0.5 is to give  the r^2 kernel, not
                               ; the r kernel

    FOR i =  0, nr-1 DO BEGIN
        FOR j = 0, i DO BEGIN
            te = 0.5 * sqrt (rad(i)^2 + rad(j)^2 - (2 * rad(i) * rad(j))*cth)
                               ; te in units of the diameter, not the radius
            te = call_function (stf, te)
            kelt =  fnorm * dth * float (fft(te, 1))
            kers (i, j, *) = kelt
            kers (j, i, *) = kelt
        END
        ;print, i, format = '(i3, $)'
    END
    ;print, ' '
    return, kers
END




FUNCTION  piston_orth, nr
    s = fltarr (nr, nr)
    FOR j = 0, nr-2 DO BEGIN
        rnm = 1. / sqrt (float ((j+1) * (j+2)))
        s (0:j, j) = rnm
        s(j+1,j) =  -1 *  (j+1) * rnm
    END
    rnm = 1. / sqrt (nr)
    s(*, nr-1) = rnm
    return, s
END




PRO gkl_fcom, kers, nf, ri,  evals, nord, npo, ord, rabas

    s = size (kers) & nr = s(1) &  nt = s(3)
    nxt = 0
    fktom =  (1. - ri^2) / nr  ; this is the one I can't get right!!!!
    fevtos = sqrt (2 * nr)
    evs = fltarr (nr, nt)

                               ; ff isnt used - the normalisation for
                               ; the eigenvectors is straightforward:
                               ; integral of surface^2 divided by area = 1,
                               ; and the cos^2 term gives a factor
                               ; half, so multiply zero order by
                               ; sqrt(n) and the rest by sqrt (2n)
; zero order is a special case...
; need to deflate to eliminate infinite eigenvalue - actually want
; evals/evecs of zom - b where b is big and negative
    zom = kers (*,*,0)
    s = piston_orth (nr)
    b1 = (transpose (s) # zom # s) (0:nr-2, 0:nr-2)

    newev = eig (fktom * b1, vectors = v0, /symmetric)
    v1 = fltarr (nr, nr)
    v1(0,0) = v0
    v1(nr-1,nr-1) = 1
    vs = s # v1
    newev = [newev, 0]
    evs(*, nxt) = newev
    kers (*, *, nxt) = sqrt(nr) * vs

; the rest are more straightforward
    nxt = 1
    REPEAT BEGIN
        newev = eig (fktom * kers(*, *, nxt), vectors = vs, /symmetric)
        evs(*, nxt) = newev
        kers (*, *, nxt) = sqrt(2. * nr) * vs
        ;print, nxt, format = '(i4, $)'
        mxn = max (newev)
        egtmxn = fix (evs(*, 0:nxt) GT mxn)
        nxt = nxt + 1
    END UNTIL  (2 * total (egtmxn) - total (egtmxn(*,0))) GE nf
    nus = nxt - 1
    ;print, ' '

    kers = kers (*, *, 0:nus-1)
    evs = reform (evs (*, 0:nus-1), nr*nus)
    a = (sort (-1. * evs))(0:nf-1)
                               ; every eigenvalue occurs twice except
                               ; those for the zeroth order mode. This
                               ; could be done without the loops, but
                               ; it isn't the sticking point anyway...
    no = 0
    ni = 0
    oind = lonarr(nf+1)
    REPEAT BEGIN
       IF a(ni) LT nr THEN BEGIN
           oind(no) = a(ni)
           no = no + 1
       END ELSE BEGIN
           oind(no) = a(ni)
           oind(no+1) = a(ni)
           no = no + 2
       END
       ni = ni + 1
    END UNTIL no GE nf

    oind = oind (0:nf-1)
    tord = oind / nr
    odd = ((lindgen(nf) MOD 2) EQ 1)
    pio = oind MOD nr


    evals = evs(oind)
    ord = 2 * tord - fix (tord GE 1 AND odd)
    nord = max (ord) + 1
    rabas = fltarr (nr, nf)
    npo = intarr (max (ord) + 1)

    FOR i = 0, nf-1 DO BEGIN
        npo(ord(i)) = npo(ord(i)) + 1
        rabas(*, i) = kers (*, pio(i), tord(i))
    END
END


FUNCTION gkl_mkazi, nord, np

    gklazi = fltarr (1 + nord, np)
    th = findgen (np) * (2. * !pi / np)

    gklazi (0, *) = 1.0
    FOR i = 1, nord-1, 2  DO  gklazi (i, *) = cos ((i/2+1) * th)
    FOR i = 2, nord-1, 2  DO  gklazi (i, *) = sin ((i/2) * th)
    return, gklazi
END




FUNCTION gkl_bas, ri=ri, nr=nr, np=np, nfunc=nfunc, stf=stf

    IF n_elements(ri) eq 0     THEN ri = 0.25
    IF NOT keyword_set (nr)    THEN nr = 40L
    IF NOT keyword_set (np)    THEN np = long(5*nr)
    IF NOT keyword_set (nfunc) THEN nfunc = 500L
    IF NOT keyword_set (stf)   THEN stf = 'kolstf'

    nr = long(nr)
    np = long(np)

    IF  (nr * np) / nfunc LT 8 THEN BEGIN
        print, "warning: you may need a finer radial sampling "
        print, "(ie, increased 'nr') to generate ", nfunc, "  functions"
    END ELSE IF (nr * np) / nfunc GT 40 THEN BEGIN
        print, "note, for this size basis, radial discretization on ", nr
        print, "points is finer than necessary - it should work, but you "
        print, "could take a smaller 'nr' without loss of accuracy"
    END


    radp = gkl_radii (nr, ri)

    kers = gkl_mkker (ri, nr, radp, stf)

    gkl_fcom, kers, nfunc, ri, evals, nord, npo, ord, rabas

    azbas = gkl_mkazi (nord, np)

    gklbasis = {$
                 nr:nr, np:np, nfunc:nfunc, $
                 ri:ri, stfn:' ',  $
                 radp:radp, $
                 evals:evals, nord:nord, npo:npo, ord:ord,  $
                 rabas:rabas, azbas:azbas $
                 }
    return, gklbasis
END





