;+ Author: Laura Schreiber - August 2022
;
; ROUTINE NAME:
;SEQ_asong_slope
;
;PURPOSE:
;Given a frame and the binary masks defining the four quadrants of the ASONG WFS, compute the wavefront derivatives in X and Y.
;Modification of the asong_slope procedure, when the images of the 4 pupils are taken in sequence. 
;
;CALLING SEQUENCE:
;asong_slope, frame, qmask, pup, centres, sx, sy
;
;INPUT:
;frame:  cube of measured frames with subtracted background and divided by the flat (if required)
;qmask:  3D array with four planes, each representing a binary mask 
;        associated to one quadrant of the sensor
;pup:    footprint of sx and sy
;centres: 4 X 2-elements vector of coordinates of pupils centroids
;
;KEYWORDS:
;gain:        when this keyword is set, the gain is not computed and the value set in the keyword is used instead.
;             It can come from calibration procedures. DEFAULT VALUE = 0
;pyr:         Set this keyword in case a Pyramid WFS is used in stead of ASONG. DEFAULT VALUE = 0    
;dxf, dyf:    GTF x and y period in mm. If only one of the two keywords is set, the period in X and Y is considered to be the same. 
;             DEFAULT VALUE = 9.05 mm  
;foc:         distance from the aperture and the fourier plane
                       
;
;OUTPUT:
;sx, sy: array of signals in X and Y

;-


PRO SEQ_asong_slope, frame, qmask, pup, centres, $                            ;inputs
                  GAIN = GAIN, dxf = dxf, dyf = dyf, foc = foc, $             ;keywords
                  sx, sy                                                      ;outputs


;on_error, 2

if not keyword_set(PYR) then begin
   if not keyword_set(foc) then foc = 500.  ; focal lengh in mm
   if not keyword_set(dxf) then dxf = 9.05    ; period in mm
   dyf = dxf
   gx = dxf / foc / 2. / !dpi / sqrt(2.)
   gy = dyf / foc / 2. / !dpi / sqrt(2.)
endif else begin 
   if not keyword_set(GAIN) then begin
      gx = 1.0 & gy = 1.0  
   endif else begin
      gx = gain & gy = gain
   endelse
endelse


dimpup = (size(pup,/dim))[0]
c=centres
c = transpose(c)
; Extract four quadrants from the frame
nquad = 4
quad = fltarr(dimpup, dimpup, nquad)
lx = round(c[*,0]) - dimpup/2
ux = round(c[*,0]) + dimpup/2 
ly = round(c[*,1]) - dimpup/2
uy = round(c[*,1]) + dimpup/2
for q = 0, nquad - 1 do begin
   quad[*,*,q] = (frame[*,*,q] * qmask[*,*,q])[lx:ux-1,ly:uy-1]*pup
endfor

; Compute signal and derivative
sx = fltarr(dimpup, dimpup)
sy = fltarr(dimpup, dimpup)
norm = total(quad, 3)
quadmask = (pup and (norm ne 0)) and 1B
w = where(quadmask)

if keyword_set(PYR) then begin
  sx[w] = ((quad[*,*,3] + quad[*,*,1]) - (quad[*,*,2] + quad[*,*,0]))[w]
  sy[w] = ((quad[*,*,2] + quad[*,*,3]) - (quad[*,*,0] + quad[*,*,1]))[w]
  sx[w] = gx * sx[w] / norm[w]
  sy[w] = gy * sy[w] / norm[w]  
endif else begin
  sx[w] = ((quad[*,*,3] + quad[*,*,1]) - (quad[*,*,2] + quad[*,*,0]))[w]/visx
  sy[w] = ((quad[*,*,0] + quad[*,*,1]) - (quad[*,*,2] + quad[*,*,3]))[w]/visy
  sx[w] = sx[w] / norm[w]
  sy[w] = sy[w] / norm[w]  
  sx[w] = asin(sx[w]) * gx
  sy[w] = asin(sy[w]) * gy  
endelse


return
end
