;+ Author: Laura Schreiber - August 2022
;
;ROUTINE NAME:
;asong_image
;
;PURPOSE:
;Average and eventually background subtract data images
;
;CALLING SEQUENCE:
;asong_image, path, bkg=bkg, ima_avg
;
;INPUT:
;path: a string containing the folder where the images are stored 
;
;KEYWORDS:
;bkg = bkg : with this keyword set, the background given as input is    
;            subtracted when the bkg is not given but the keyword is set, the median of the
;            image is subtracted
;bmp = bmp : set this keyword to read bmp files. Default format is .tiff extension
;save = save : set this keyword to save the average image in the same path
;name = name : set this keyword to change the default value = 'avg_ima'
;jpeg = jpeg : set this keyword to save the image in jpeg format. The default value is bmp (and sav format)  
;dimx = dimx : set this keyword to set the x size of the acquired image. Default value: 3072
;dimy = dimy : set this keyword to set the y size of the acquired image. Default value : 2048          
;
;OUTPUT:
;ima_avg :  the average image computed on the n_ima taken images 

;example 
; path = 'folder path here'
; asong_image,path,/bkg,avg_test,/save,name = 'test'


pro asong_image, path,          $            ;inputs
                 bkg  = bkg,    $
                 bmp  = bmp,    $
                 save = save,   $
                 name = name,   $
                 jpeg = jpeg,   $            ;keywords
                 dimx = dimx,   $
                 dimy = dimy,   $
                 ima_avg                     ;outputs
             
           
           
           
           
           on_error,2
           
           
           if not keyword_set(dimx) then dimx = 3072 
           if not keyword_set(dimy) then dimy = 2048
           
           if not keyword_set(name) then name = 'avg_ima'
           if keyword_set(bmp) then begin
              images = FILE_SEARCH(Path + '*.bmp', count = n_ima)
           endif else begin
             images = FILE_SEARCH(Path + '*.tiff', count = n_ima)
           endelse
           ima_avg  = fltarr(dimx,dimy)
          
           imax=0.0

           for i=0, n_ima-1 do begin
             if keyword_set(bmp) then ima = read_bmp(images[i]) else $
             ima = read_tiff(images[i])

             if max(ima) gt imax then imax=max(ima)

             if keyword_set(bkg) then begin
               if (size(bkg))[0] eq 0 then begin
                 bkg = median(ima)
               endif
               ima = ima - bkg
               ima_avg = ima_avg + ima
             endif else begin
               ima_avg = ima_avg + ima
             endelse
           endfor
                     
           ima_avg = ima_avg / n_ima
           ima_avg = ima_avg > 0.
           
         
           if imax eq 255 then print,'image saturated' else print,"MAX intensity per frame = ", imax
           if keyword_set(save) then begin
            save,ima_avg,filename = path + name + '.sav'
            LOADCT, ecol
            WRITE_BMP, path + name + '.bmp', BYTSCL(avg_ima)
            LOADCT, 0
            if keyword_set(jpeg) then begin
              window, xs = dimx/2, ys = dimy/2
              tvscl, congrid(ima_avg,dimx/2,dimy/2)
              write_jpeg,path + name + '.jpg', tvrd()
            endif
           endif
               
           

return
end             
             