;+ Author: Laura Schreiber - August 2022
;
; ROUTINE NAME:
; asong_pupil_edge_fit
; 
; PURPOSE:
; fit a pupil image with a circle or an ellipse
;
; INPUTS:
; ima: 2D array with pupil image
; 
; KEYWORDS
; [/threshold]: relative threshold for edge detection. The value is between 0 and 1. Default value = 0.1
; [/CIRCULAR]: set this keyword to fit a circle instead of an ellipse
; [/AUTO] : set this keyword to fit the edge of the pupils using a threshold and sobel algorithm. If not set, click on the image edge will be necessary. 
; [/LOG]  : set this keyword to bust the contrast of the image
; wscale    = wscale    :  Variable type: Scalar. Graphycal window scaling factor w.r.t. image size. DEFAULT VALUE = 0.5
;
; OUTPUTS:
; centres: pupil centres in the format compatible with asong_mask procedure
; p_all: vector of ellipse parameters:
;    p_all[0], p_all[1]: semi-axes
;    p_all[2], p_all[3]: center X and Y coordinates
;    p_all[4]: tilt angle of the ellipse
; x_cen, y_cen: vectors of coordinates of points fitted by ellipse
; rad : estimated pupils radii
;
; PROCEDURE:
; Mask image above given relative threshold. Apply SOBEL filter to 
; binary mask to find edge. Fit edge by MPFITELLIPSE (see MPFIT package).
;
;-

PRO asong_pupil_edge_fit, ima,                   $    ;input
                          threshold = threshold, $    ;keywords
                          CIRCULAR = circular,   $
                          quad = quad,           $
                          wscale = wscale,       $
                          AUTO = AUTO,           $
                          log = log,             $
                          _EXTRA = extra,        $ 
                          centres, p_all, x_cen, y_cen, rad      ;outputs
 
on_error,2

if not keyword_set(log) then log = 0
dimx = float((size(ima, /DIM))[0])
dimy = float((size(ima, /DIM))[1])
if not keyword_set(wscale)  then wscale  =  0.5
xw_size = wscale * dimx
yw_size = wscale * dimy

if log eq 1 then log = 0.1

y_coord=rebin(transpose(findgen(dimy)),dimx,dimy)
x_coord=rebin(findgen(dimx),dimx,dimy)

mask_preview = fltarr(dimx,dimy)
circ_preview = fltarr(dimx,dimy)

if keyword_set(quad) then begin
  n_pup = 4 
  endif else begin
  n_pup = 1
  quad = fltarr((size(ima, /DIM))[0],(size(ima, /DIM))[1])+1
endelse

p_all = fltarr(5,n_pup)
x_cen = fltarr(n_pup) &  y_cen = fltarr(n_pup)
edge_all = fltarr(dimx, dimy, n_pup)
rad = fltarr(n_pup)

;-------------------------------------
for i = 0,n_pup -1 do begin 

    imai = ima * quad[*,*,i]
  if keyword_set(auto) then begin 
    if not keyword_set(threshold) then threshold = 0.1 
    mask = imai gt threshold * max(imai)
    mask = median(mask, 3)
    ;mask = smooth(float(mask), 3)

    m = max(mask, w)
    x = w mod dimx  &  y = w / dimx
    w = search2d(mask, x, y, 1, 1)
    x = w mod dimx  &  y = w / dimx
    dx = max(x) - min(x)
    dy = max(y) - min(y)
    d = (dx + dy) / 2.0
    cx = (min(x) + max(x)) / 2.0
    cy = (min(y) + max(y)) / 2.0

    edge = sobel(mask)
    w = where(edge)
    x = w mod dimx
    y = w / dimx


    p = [d/2.0, d/2.0, cx, cy, 0.0]
    we = edge[x, y]
    edge_all[*,*,i] = edge
    window,5,title='edges preview', xs = xw_size, ys = yw_size
    tvscl, congrid(total(edge_all,3),xw_size,yw_size)
  endif else begin
  
    window,title = 'Select at least three points of the mask edge ', xs = xw_size, ys = yw_size
    if not keyword_set(log) then begin
      tvscl, congrid(imai,xw_size,yw_size)
    endif else begin
      tvscl, congrid(imai^log,xw_size,yw_size)
    endelse
    
    if  not keyword_set(silent)  then begin
      print, 'Select at least three points of the mask edge by clicking with the left button of your mouse...'
       print, 'Push right button to exit'
    endif
    edge = fltarr(dimx, dimy)
    !MOUSE.button = 1  &  n_click = 0
    cursor, x_click, y_click, /NORMAL, /DOWN
    while  !MOUSE.button eq 1  do begin
    
      x_click = round(x_click * dimx)  &  y_click = round(y_click * dimy)
      if  n_elements(x_saved) eq 0  then begin
        x_saved = x_click
        y_saved = y_click
      endif else begin
        x_saved = [x_saved, x_click]
        y_saved = [y_saved, y_click]
      endelse
      crosses, tvrd(), x_click/(dimx - 1)*!D.x_size, y_click/(dimy - 1)*!D.y_size, $
        /EXISTING, _EXTRA = extra, /DEVICE, symsize = 3
      if  !MOUSE.button eq 1  then  cursor, x_click, y_click, /NORMAL, /DOWN
    endwhile
  if  n_elements(x_saved) ne 0  then begin
    x = x_saved  &  y = y_saved   
  endif
   tempvar = SIZE(TEMPORARY(x_saved))
   tempvar = SIZE(TEMPORARY(y_saved))
  ; undefine,x_saved
  ; undefine,y_saved
  ;asong_circle,x[0:2],y[0:2],cx,cy,r
  cx = total(x)/n_elements(x) & cy = total(y)/n_elements(y)
  r = avg(sqrt((x-cx)^2+(y-cy)^2))
  we = fltarr(n_elements(x))+0.75
  p = [r, r, cx, cy, 0.0]
   ; if  n_click eq 1 then  !MOUSE.button = 4
  endelse
    p = mpfitellipse(float(x), float(y), p, /TILT, CIRCULAR = circular, WEIGHTS = we, /QUIET)
;    if p[0] lt p[1] then begin
  ;     temp = p[0]
  ;     p[0] = p[1]
  ;     p[1] = temp
  ;     p[4] = p[4] + !pi/2
  ;  endif
  p[4] = (p[4] / !pi * 180) mod 360
  p[4] = 360 - p[4]
  p[4] = p[4] mod 180
  p_all[*,i] = p
 
  x_cen[i] = p[2]
  y_cen[i] = p[3]
  rad[i] = p[0]
   
  mask_preview=mask_preview + float(sqrt((x_coord-x_cen[i])^2+(y_coord-y_cen[i])^2) le rad[i])

  

endfor
centres = [[x_cen],[y_cen]]


window,0,title='mask preview', xs = xw_size, ys = yw_size
tvscl, congrid(mask_preview,xw_size,yw_size)
window,7, title = 'quick_check', xs = xw_size, ys = yw_size
tvscl, congrid(ima,xw_size,yw_size)
points = (2 * !PI / 3999.0) * FINDGEN(4000)
scale = [!D.x_size, !D.y_size] / float(size52(ima, /DIM) - 1)
for i = 0,3 do plots,[(x_cen[i] + rad[i] * COS(points))]*scale[0],[(y_cen[i] + rad[i] * SIN(points))]*scale[1],/device,psym = 3

;tvscl, congrid(circ_preview,xw_size,yw_size)
print,'the coordinates of the pupils centers are: '
print, 'x = [' + strcompress(string(x_cen[0])) + ','+ strcompress(string(x_cen[1]))+ ',' + strcompress(string(x_cen[2]))+ ',' + strcompress(string(x_cen[3])) + ' ]'
print, 'y = [' + strcompress(string(y_cen[0])) + ','+ strcompress(string(y_cen[1]))+ ',' + strcompress(string(y_cen[2]))+ ',' + strcompress(string(y_cen[3])) + ' ]'
print,'the 4 pupils radii are: '
print, 'rad = [' + strcompress(string(rad[0])) + ','+ strcompress(string(rad[1]))+ ',' + strcompress(string(rad[2]))+ ',' + strcompress(string(rad[3])) + ' ]'
return
end
