;+ Author: Laura Schreiber - August 2022
;
; ROUTINE NAME:
;asong_mask
;
;PURPOSE:
;Given a the centres of the four pupils, builds the binary mask for the signal computation and the pupil mask
;
;CALLING SEQUENCE:
;asong_mask, centres, px = px, dim = dim , dimpup = dimpup, rad = rad, qmask, pup, x, y, opup, ox, oy, odimpup
;
;INPUT:
;centres: 4 X 2-elements vector of coordinates of metapupils centroids
;
;KEYWORDS:
;px = px :   oversampling factor for the numerical derivative 
;            computation in lz_derivative. Default value is 1
;dim = dim : The dimension of the frame containing the pupils images.
;            Default value is 3072X2048 
;dimpup = dimpup: The dimension of the array that will contain the 
;                 pupil for the wavefront calculation. It can be 
;                 slightly larger than the calculated pupil. Default
;                 value is 900
;rad = rad :    Measured pupil radius. Default value is 450 pixels     
;
;OUTPUT:
;qmask:  3D array with four planes, each representing a binary mask
;        associated to one quadrant of the sensor
;pup:    pupil mask, logical AND of four masks
;x, y:   coordinates
;opup:   pupil mask defined on a finer grid (defined by the px keyword)
;ox, oy: coordinates in the finer reference
;odimpup:dimension of opup

;-
;MODIFICATION HISTORY:
; 29/09/2022: added +1 to the rad value in the qmask definition to avoid edge issues

PRO asong_mask, centres,                $                                                            ;input
                px        = px        , $
                dimension = dimension , $
                dimpup    = dimpup    , $
                rad       = rad       , $   
                _EXTRA    = EXTRA     , $    ;keywords  
                qmask, pup, x, y, opup, ox, oy, odimpup, c                            ;output


on_error,2
if (size(centres))[0] eq 1 then c = transpose(rebin(centres,2,4)) else $
c=centres

if not keyword_set(rad) then rad=450
if not keyword_set(dimension) then dimension=[3072,2048]
if not keyword_set(dimpup) then dimpup = 2 * rad
if not keyword_set(px) then px=1

qmask=fltarr(dimension[0],dimension[1],4)

y=rebin(transpose(findgen(dimension[1])),dimension[0],dimension[1])
x=rebin(findgen(dimension[0]),dimension[0],dimension[1])

for i=0,3 do begin
   qmask[*,*,i]=sqrt((x-c[i,0])^2+(y-c[i,1])^2) le rad + 1
endfor

y=rebin(transpose(findgen(dimpup)),dimpup,dimpup)-dimpup/2
x=rebin(findgen(dimpup),dimpup,dimpup)-dimpup/2
pup=sqrt(x^2+y^2) le rad

odimpup = px * dimpup
oy=rebin(transpose(findgen(odimpup)),odimpup,odimpup)-odimpup/2
ox=rebin(findgen(odimpup),odimpup,odimpup)-odimpup/2

d = odimpup + 2
oy = rebin(transpose(findgen(d)),d,d)-d/2
ox = rebin(findgen(d),d,d)-d/2
opup = sqrt(ox^2+oy^2) le rad * px


return
end