;+ Author: Laura Schreiber - August 2022
;
; ROUTINE NAME:
;asong_slope
;
;PURPOSE:
;Given a frame and the binary masks defining the four quadrants of the ASONG WFS, compute the wavefront derivatives in X and Y.
;
;CALLING SEQUENCE:
;asong_slope, frame, qmask, pup, centres, sx, sy
;
;INPUT:
;image:  measured frame with subtracted background and dovided by the flat (if required)
;qmask:  3D array with four planes, each representing a binary mask 
;        associated to one quadrant of the sensor
;pup:    footprint of sx and sy
;centres: 4 X 2-elements vector of coordinates of pupils centroids
;
;KEYWORDS:
;gain:        when this keyword is set, the gain is not computed and the value set in the keyword is used instead.
;             It can come from calibration procedures. DEFAULT VALUE = 0
;pyr:         Set this keyword in case a Pyramid WFS is used in stead of ASONG. DEFAULT VALUE = 0    
;dxf, dyf:    GTF x and y period in mm. If only one of the two keywords is set, the period in X and Y is considered to be the same. 
;             DEFAULT VALUE = 9.05 mm  
;foc:         distance from the aperture and the fourier plane
;display:     set this keyword to visualise the computed slopes in a window
;pup_scale:   System magnification. Theoretical value is given by the ratio between the focalising optic focal length and
;             and the MLA focal length. This parameter should be measured in the lab by reimaging in the pupil an object of known lenght             
;det_px:      detector pixel size in microns. DEFAULT VALUE = 2.4                        
;
;OUTPUT:
;sx, sy: array of signals in X and Y

;-
; NOTES: this procedure is adapted for the MLA with 4 lenses. In case of more lenses, minor modifications needs to be done in the code
;
;dir = 'C:\Users\schreibl\IDLWorkspace\Default\ASONG_software\TEST\'  -- directory of your test folder
;name = 'test_'
;restore, dir + 'ASONG_mask_test.sav'
;restore,dir + 'test_image.sav'
;asong_slope, ima_avg, qmask, pup, centres, sx, sy ,/disp,/save, dir = dir, name = name

PRO asong_slope, image, qmask, pup, centres, $                               ;inputs
                 GAIN = GAIN               , $
                 dxf = dxf                 , $ 
                 dyf = dyf                 , $ 
                 foc = foc                 , $ 
                 display = display         , $ 
                 pup_scale = pup_scale     , $ 
                 det_px = det_px           , $ 
                 save_file = save_file     , $
                 dir = dir                 , $
                 wscale = wscale           , $
                 name = name               , $            ;keywords
                 sx, sy                                                      ;outputs


;on_error, 2
 
if keyword_set(save_file) and not keyword_set(dir) then begin
  print, 'files will be saved in your working directory'
  pwd
endif
if not keyword_set(wscale) then wscale = 1
   if not keyword_set(name)      then name = ''
if not keyword_set(PYR)          then begin
   if not keyword_set(foc)       then foc = 500.  ; focal lengh in mm
   if not keyword_set(dxf)       then dxf = 9.05    ; period in mm
   if not keyword_set(pup_scale) then pup_scale = 12.10 ; pupil object to pupil image scale factor
   if not keyword_set(det_px)    then det_px = 2.4         ; detector pixel size in microns

   
   dyf = dxf
   gx = dxf / foc / 2. / !dpi / sqrt(2.) * (det_px * pup_scale)
   gy = dyf / foc / 2. / !dpi / sqrt(2.) * (det_px * pup_scale)
endif else begin 
   if not keyword_set(GAIN) then begin
      gx = 1.0 & gy = 1.0  
   endif else begin
      gx = gain & gy = gain
   endelse
endelse


dimpup = (size(pup,/dim))[0]
c=centres

; Extract four quadrants from the frame
nquad = 4
quad = fltarr(dimpup, dimpup, nquad)
lx = round(c[*,0]) - dimpup/2
ux = round(c[*,0]) + dimpup/2 
ly = round(c[*,1]) - dimpup/2
uy = round(c[*,1]) + dimpup/2
for q = 0, nquad - 1 do begin
      quad[*,*,q] = (image * qmask[*,*,q])[lx[q]:ux[q]-1,ly[q]:uy[q]-1]*pup
endfor

; Compute signal and derivative
sx = fltarr(dimpup, dimpup)
sy = fltarr(dimpup, dimpup)
norm = total(quad, 3)
quadmask = (pup and (norm ne 0)) and 1B
w = where(quadmask)

if keyword_set(PYR) then begin
  sx[w] = ((quad[*,*,3] + quad[*,*,1]) - (quad[*,*,2] + quad[*,*,0]))[w]
  sy[w] = ((quad[*,*,2] + quad[*,*,3]) - (quad[*,*,0] + quad[*,*,1]))[w]
  sx[w] = gx * sx[w] / norm[w]
  sy[w] = gy * sy[w] / norm[w]  
endif else begin
  sx[w] = ((quad[*,*,3] + quad[*,*,1]) - (quad[*,*,2] + quad[*,*,0]))[w];/visx
  sy[w] = ((quad[*,*,0] + quad[*,*,1]) - (quad[*,*,2] + quad[*,*,3]))[w];/visy
  sx[w] = sx[w] / norm[w]
  sy[w] = sy[w] / norm[w]  
  if total(where(abs(sx[w]) gt 1.)) gt 0 then begin
    print,'WARNING! Detected' + strcompress(string(n_elements(where(abs(sx[w]) gt 1.))))+ ' infinite values in the slope computation'
    print,'Automathic sobstitution of infinite values...'
    sx[where(abs(sx*quadmask) gt 1.)] = signum((sx[w])[where(abs(sx[w]) gt 1.)])
  endif
  if total(where(abs(sy[w]) gt 1.)) gt 0 then begin
    print,'WARNING! Detected' + strcompress(string(n_elements(where(abs(sy[w]) gt 1.))))+ ' infinite values in the slope computation'
    print,'Automathic sobstitution of infinite values...'
    sy[where(abs(sy*quadmask) gt 1.)] = signum((sy[w])[where(abs(sy[w]) gt 1.)])
  endif
  sx[w] = asin(sx[w]) * gx
  sy[w] = asin(sy[w]) * gy  
endelse

if keyword_set(display) then begin
  window, 0 , title = 'Slope X and Y', xs = wscale * dimpup*2, ys = wscale * dimpup
  tvscl,congrid([sx,sy],wscale *dimpup*2,wscale *dimpup)
endif

if keyword_set(save_file) then begin
  if keyword_set(dir) then begin
    save, sx, sy, pup, filename = dir + 'ASONG_slopes' +name + '.sav' 
   
      WRITE_BMP, dir + 'ASONG_slopes' + name + '_preview.bmp', BYTSCL([sx,sy])
  
  endif else begin
    save, sx, sy, pup, filename = 'ASONG_slopes' +name + '.sav' 
    WRITE_BMP,'ASONG_slopes' + name + '_preview.bmp', BYTSCL([sx,sy])
  endelse
endif


return
end
