;+ Author: Laura Schreiber - August 2022
;
; ROUTINE NAME:
;asong_mask_main
;
;PURPOSE:
;Given an image of the four pupils, it is used to define the quadrant mask for slope computation
;
;CALLING SEQUENCE:
;asong_mask_main, image, qmask, pup, centres
;
;INPUT:
;image :     2D array containing the image of the four pupils.
;
;OPTIONAL INPUT:
;centres : when the keyword /CEN is set, a 2D array containing the centres of the four pupils in the correct order can be passed as an input 
;
;KEYWORDS:
;save_file = save_file :  Variable type: LOGICAL. Use this keyword to save the mask and the output variables 
;dir = dir             :  Variable type: STRING. Set this keyword to define the directory in which the variables must be stored. 
;name = name           :  Variable type: STRING. Set this keyword to set the name of the file. DEFAULT VALUE = 'ASONG_mask'
;dimpup = dimpup       :  Variable type: SCALAR. The dimension of the array that will contain the
;                         pupil for the wavefront calculation. It can be
;                         slightly larger than the calculated pupil. Default value is 900
;log = log             :  Variable type: LOGICAL. Set this keyword to display more contrasted images to help some cliccking operation
;auto = auto           :  Variable type: LOGICAL. Set this keyword to use SOBEL algorithm to find the pupils edges. 
;                                      WARNING!!!! I recommand to use this keyword only in case light distribution in the 4 pupils is homogeneus
;threshold = threshold :  Variable type: Scalar, float. Normalised value (less than 1). Set the threshold for the automatic mask definition. DEFAULT VALUE = 0.1 of the max
;n_pupils  = n_pupils  :  Variable type: Scalar, integer. DEFAULT VALUE = 4. Number of pupils in the image. 1 and 4 are acceptables
;wscale    = wscale    :  Variable type: Scalar. Graphycal window scaling factor w.r.t. image size. DEFAULT VALUE = 0.5
;silent    = silent    :  Variable type: LOGICAL. Set this keyword to avoid unecessary output in the IDL Console. 
;dimpup    = dimpup    :  Variable type: Scalar. DEFAUKT VALUE = 900 image pixels. The dimension of the array that will contain the
;                                                                    pupil for the wavefront calculation. It can be
;                                                                    slightly larger than the calculated pupil.
;rad       = rad       :  Variable type: Scalar. DEFAULT VALUE = 450 image pixels. Pupil mask radius
;cen       = cen       :  Variable type: LOGICAL. If set, the centres variable is used as an input variable and the centers coordinates are used to compute the mask
;
;OUTPUT:
;qmask:  3D array with four planes, each representing a binary mask
;        associated to one quadrant of the sensor
;pup:    pupil mask, logical AND of four masks
;centres : 4 pupil centres

;EXAMPLE
;dir = 'C:\Users\schreibl\IDLWorkspace\ASONG_software_v1.0\TEST\'
;name = 'test_'
;restore,dir + 'test_image.sav'
;asong_mask_MAIN, ima_avg, /log, /save, dir = dir, name = name, centres, qmask, pup 


PRO asong_mask_MAIN, image                 , $
                     save_file = save_file , $
                     dir       = dir       , $
                     name      = name      , $
                     log       = log       , $
                     threshold = threshold , $                  
                     n_pupils  = n_pupils  , $
                     wscale    = wscale    , $
                     AUTO      = AUTO      , $                     
                     silent    = silent    , $
                     dimpup    = dimpup    , $
                     rad       = rad       , $
                     cen       = cen       , $
                     _EXTRA    = extra     , $
                     centres, qmask, pup   




if not keyword_set(name)      then    name      = ''
if not keyword_set(threshold) then    threshold = 0.1
if not keyword_set(n_pupils)  then    n_pupils  = 4
if not keyword_set(wscale)    then     wscale   =  0.5
if not keyword_set(dimension) then    dimension = [(size(image,/dim))[0],(size(image,/dim))[1]]
if not keyword_set(rad)       then    rad = 450
if not keyword_set(dimpup)    then    dimpup    = 2*rad

if keyword_set(save_file) and not keyword_set(dir) then begin
  print, 'files will be saved in your working directory'
  pwd
endif

if not keyword_set(cen) then begin
   if n_pupils eq 4 then begin

    quad = asong_define4quad(image,             $                
                           wscale = wscale,   $
                           silent = silent,   $
                           log = log,         $
                           _EXTRA = extra)  
  endif

; Pupil mask definition
  asong_pupil_edge_fit, image, threshold = threshold, $    ;keywords
                             CIRCULAR = 1,          $
                             quad = quad,           $
                             wscale = wscale,       $                             
                             AUTO = AUTO,           $
                             log = log,             $
                             _EXTRA = extra,        $
                             centres 
endif else begin
  if n_elements(centres) eq 0 then print,'Center coordinate not defined... returning'
  return
endelse

; Given the pupil centres, define the pupil masks 
asong_mask, centres                                                    , $
            dimension = [(size(image,/dim))[0],(size(image,/dim))[1]]  , $
            dimpup    = dimpup                                         , $
            rad       = rad                                            , $
            _EXTRA    = EXTRA                                          , $    ;keywords
            qmask, pup

if not keyword_set(silent) then print,'Used pupil radius = ' +strcompress(string(rad),/rem)
; check mask alignement
asong_check_mask, image, qmask         , $
                      wscale  = wscale , $
                      log     = log    , $
                      _EXTRA  = extra    ;keyword

; saving
if keyword_set(save_file) then begin
  if n_elements(size(qmask,/dim)) eq 3 then $
    mask = total(qmask,3)  else $
    mask = qmask
  if keyword_set(dir) then begin
    save, qmask, pup, centres, rad, dimpup, filename = dir + 'ASONG_mask' + name + '.sav' 
     WRITE_BMP, dir + 'ASONG_mask' + name + '_preview.bmp', BYTSCL(mask)
  endif else begin
    save, qmask, pup, centres, rad, dimpup,  filename = 'ASONG_mask' + name + '.sav' 
    WRITE_BMP,'ASONG_mask' + name + '_preview.bmp', BYTSCL(mask)
  endelse
endif

return
end