#include <proxy.hpp>
#include <iomanip>

namespace atk_cli
{
    
//==============================================================================
//  proxy::get_state()
//==============================================================================
void proxy::get_state()
{
    try
    {
        Tango::DevState state = device_proxy_p->state();
        
        std::cout << "State: " << state_to_string(state) << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
    catch(std::exception& ex)
    {
        std::cerr << ex.what() << std::endl;
    }
}

//==============================================================================
//  proxy::get_status()
//==============================================================================
void proxy::get_status()
{
    try
    {
        std::string status = device_proxy_p->status();
                
        std::cout << "Status: " << status << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::list_attributes()
//==============================================================================
void proxy::list_attributes()
{
    try
    {
        Tango::AttributeInfoListEx* attr_list
            = device_proxy_p->attribute_list_query_ex();
        
        int count = 0;
        Tango::AttributeInfoListEx::iterator it;
        
        for(it=attr_list->begin(); it!=attr_list->end(); ++it)
        {
            std::cout << "[" << count++ << "] " << it->name << std::endl;
        }
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::read_attribute()
//==============================================================================
void proxy::read_attribute(std::string attr_name)
{
    try
    {
        Tango::DeviceAttribute dev_attr
            = device_proxy_p->read_attribute(attr_name);
                
        std::cout << "Attribute: " << attr_to_string(dev_attr) << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
    catch(std::exception& ex)
    {
        std::cerr << ex.what() << std::endl;
    }    
}

//==============================================================================
//  proxy::state_to_string()
//==============================================================================
std::string proxy::state_to_string(Tango::DevState state)
    throw(std::invalid_argument)
{
        if(state == Tango::ON)
            return "ON";
        else if(state == Tango::OFF)
            return "OFF";
        else if(state == Tango::CLOSE)
            return "State: CLOSE";
        else if(state == Tango::OPEN)
            return "State: OPEN";
        else if(state == Tango::INSERT)
            return "State: INSERT";        
        else if(state == Tango::EXTRACT)
            return "State: EXTRACT";        
        else if(state == Tango::MOVING)
            return "State: MOVING";        
        else if(state == Tango::STANDBY)
            return "State: STANDBY";        
        else if(state == Tango::FAULT)
            return "State: FAULT";        
        else if(state == Tango::INIT)
            return "State: INIT";        
        else if(state == Tango::RUNNING)
            return "State: RUNNING";        
        else if(state == Tango::ALARM)
            return "State: ALARM";        
        else if(state == Tango::DISABLE)
            return "State: DISABLE";        
        else if(state == Tango::UNKNOWN)
            return "State: UNKNOWN";
        else
            throw std::invalid_argument("Cannot find a known state");
}

//==============================================================================
//  proxy::attr_to_string()
//==============================================================================
std::string proxy::attr_to_string(Tango::DeviceAttribute dev_attr) 
    throw(std::invalid_argument)
{
        int type = dev_attr.get_type();
        int dim_x = dev_attr.get_dim_x();
        int dim_y = dev_attr.get_dim_y();
                
        if(dim_x>1 || dim_y>0)
            throw std::invalid_argument("Unsupported multidimensional attribute");
        
        std::stringstream string_value;
        
        switch(type)
        {
            case Tango::DEV_BOOLEAN:
            {
                Tango::DevBoolean value;
                dev_attr >> value;
                string_value << value;
                break;
            }                
            case Tango::DEV_DOUBLE:
            {
                Tango::DevDouble value;
                dev_attr >> value;
                string_value << value;
                break;   
            }   
            case Tango::DEV_FLOAT:
            {
                Tango::DevFloat value;
                dev_attr >> value;
                string_value << value;                
                break;           
            }
            case Tango::DEV_LONG:
            {
                Tango::DevLong value;
                dev_attr >> value;
                string_value << value;                
                break;                
            }    
            case Tango::DEV_LONG64:
            {
                Tango::DevLong64 value;
                dev_attr >> value;
                string_value << value;                
                break;            
            }    
            case Tango::DEV_SHORT:
            {
                Tango::DevShort value;
                dev_attr >> value;
                string_value << value;                
                break;                
            }
            case Tango::DEV_STATE:
            {
                Tango::DevState value;
                dev_attr >> value;
                string_value << state_to_string(value);
                break;           
            }    
            case Tango::DEV_STRING:
            {
                std::string value;
                dev_attr >> value;
                string_value << value;                
                break;                
            }    
            case Tango::DEV_UCHAR:
            {
                Tango::DevUChar value;
                dev_attr >> value;
                string_value << value;                
                break;                           
            }
            case Tango::DEV_ULONG:
            {
                Tango::DevULong value;
                dev_attr >> value;
                string_value << value;                
                break;                           
            }    
            case Tango::DEV_ULONG64:
            {
                Tango::DevULong64 value;
                dev_attr >> value;
                string_value << value;                                
                break;             
            }    
            case Tango::DEV_USHORT:
            {
                Tango::DevUShort value;
                dev_attr >> value;
                string_value << value;                                                
                break;            
            }                    
            case Tango::DEV_ENCODED:
                throw std::invalid_argument("Unsupported attribute type");
                
            default:
                throw std::invalid_argument("Cannot find a known state");
        }
        
        return string_value.str();
}

}   //namespace