#include <proxy.hpp>
#include <iomanip>

namespace jive_cli
{

//==============================================================================
//  proxy::list_servers()
//==============================================================================
void proxy::list_servers(bool display_details)
{
    try
    {
        std::vector<std::string> servers = db_get_server_list();

        std::cout << "Number\tServer" << std::endl;

        for(unsigned int i=0; i<servers.size(); ++i)
        {
            std::cout << "[" << i << "]\t" << servers.at(i) << std::endl;

            if(display_details)
            {
                std::vector<std::string> device_class_list =
                    db_get_device_class_list(servers.at(i));

                for(unsigned int j=0; j<device_class_list.size(); j=j+2)
                {
                    std::cout << "\t| - " << device_class_list.at(j)
                        << " - (" << device_class_list.at(j+1) << ")"
                        << std::endl;
                }
            }
        }
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::list_devices()
//==============================================================================
void proxy::list_devices(std::string server_name)
{
    if(server_name.empty())
    {
        std::cerr << "Empty server name!" << std::endl;
        return;
    }

    try
    {
        std::vector<std::string> device_class_list =
            db_get_device_class_list(server_name);

        std::cout << "Number\tServer" << std::endl;
        std::cout << "[0]\t" << server_name << std::endl;

        for(unsigned int i=0; i<device_class_list.size(); i=i+2)
        {
            std::cout << "\t| - " << device_class_list.at(i)
                << " - (" << device_class_list.at(i+1) << ")" << std::endl;
        }
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::rm_server()
//==============================================================================
void proxy::rm_server(std::string server_name)
{
    if(server_name.empty())
    {
        std::cerr << "Empty server name!" << std::endl;
        return;
    }

    try
    {
        Tango::DevString request = CORBA::string_dup(server_name.c_str());

        Tango::DeviceData argin;
        argin << request;

        device_proxy_p->command_inout("DbDeleteServer", argin);
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::rm_device()
//==============================================================================
void proxy::rm_device(std::string device_name)
{
    if(device_name.empty())
    {
        std::cerr << "Empty device name!" << std::endl;
        return;
    }

    try
    {
        Tango::DevString request = CORBA::string_dup(device_name.c_str());

        Tango::DeviceData argin;
        argin << request;

        device_proxy_p->command_inout("DbDeleteDevice", argin);
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::save_server()
//==============================================================================
void proxy::save_server(std::string server_name, std::string device_name,
    std::string class_name)
{
    if(server_name.empty())
    {
        std::cerr << "Empty server name!" << std::endl;
        return;
    }

    if(device_name.empty())
    {
        std::cerr << "Empty device name!" << std::endl;
        return;
    }

    if(class_name.empty())
    {
        std::cerr << "Empty class name!" << std::endl;
        return;
    }

    std::string dserver_name = "dserver/";
    dserver_name.append(server_name);

    try
    {
        Tango::DevVarStringArray* request = new Tango::DevVarStringArray();
        request->length(5);

        (*request)[0] = CORBA::string_dup(server_name.c_str());
        (*request)[1] = CORBA::string_dup(dserver_name.c_str());
        (*request)[2] = CORBA::string_dup("DServer");
        (*request)[3] = CORBA::string_dup(device_name.c_str());
        (*request)[4] = CORBA::string_dup(class_name.c_str());

        Tango::DeviceData argin;
        argin << request;

        device_proxy_p->command_inout("DbAddServer", argin);
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::save_device()
//==============================================================================
void proxy::save_device(std::string server_name, std::string device_name,
    std::string class_name)
{
    if(server_name.empty())
    {
        std::cerr << "Empty server name!" << std::endl;
        return;
    }

    if(device_name.empty())
    {
        std::cerr << "Empty device name!" << std::endl;
        return;
    }

    if(class_name.empty())
    {
        std::cerr << "Empty class name!" << std::endl;
        return;
    }

    try
    {
        Tango::DevVarStringArray* request = new Tango::DevVarStringArray();
        request->length(3);

        (*request)[0] = CORBA::string_dup(server_name.c_str());
        (*request)[1] = CORBA::string_dup(device_name.c_str());
        (*request)[2] = CORBA::string_dup(class_name.c_str());

        Tango::DeviceData argin;
        argin << request;

        device_proxy_p->command_inout("DbAddDevice", argin);
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::list_dev_prop()
//==============================================================================
void proxy::list_dev_prop(std::string device_name, bool display_value)
{
    if(device_name.empty())
    {
        std::cerr << "Empty device name!" << std::endl;
        return;
    }

    try
    {
        std::vector<std::string> properties =
            db_get_device_property_list(device_name);

        if(display_value)
        {
            std::cout << "Number\tProperty - Value" << std::endl;

            for(unsigned int i=0; i<properties.size(); ++i)
            {
                std::cout << "[" << i << "]\t" << properties.at(i) << " - ";

                std::vector<std::string> values =
                    db_get_device_property(device_name, properties.at(i));

                for(unsigned j=0; j<values.size(); ++j)
                    std::cout << "[" << values.at(j) << "]";

                std::cout << std::endl;
            }
        }
        else
        {
            std::cout << "Number\tProperty" << std::endl;

            for(unsigned int i=0; i<properties.size(); ++i)
            {
                std::cout << "[" << i << "]\t" << properties.at(i)
                    << std::endl;
            }
        }
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::get_dev_prop()
//==============================================================================
void proxy::get_dev_prop(std::string device_name, std::string property_name)
{
    if(device_name.empty())
    {
        std::cerr << "Empty device name!" << std::endl;
        return;
    }

    if(property_name.empty())
    {
        std::cerr << "Empty property name!" << std::endl;
        return;
    }

    try
    {
        std::cout << "Number\tProperty - Value" << std::endl;

        std::cout << "[0]\t" << property_name << " - ";

        std::vector<std::string> values =
            db_get_device_property(device_name, property_name);

        for(unsigned j=0; j<values.size(); ++j)
            std::cout << "[" << values.at(j) << "]";

        std::cout << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::rm_dev_prop()
//==============================================================================
void proxy::rm_dev_prop(std::string device_name, std::string property_name)
{
    if(device_name.empty())
    {
        std::cerr << "Empty device name!" << std::endl;
        return;
    }

    if(property_name.empty())
    {
        std::cerr << "Empty property name!" << std::endl;
        return;
    }

    try
    {
        Tango::DevVarStringArray* request = new Tango::DevVarStringArray();
        request->length(2);

        (*request)[0] = CORBA::string_dup(device_name.c_str());
        (*request)[1] = CORBA::string_dup(property_name.c_str());

        Tango::DeviceData argin;
        argin << request;

        device_proxy_p->command_inout("DbDeleteDeviceProperty", argin);
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::save_dev_prop()
//==============================================================================
void proxy::save_dev_prop(std::string device_name, std::string property_name,
    std::vector<std::string> property_list)
{
    if(device_name.empty())
    {
        std::cerr << "Empty device name!" << std::endl;
        return;
    }

    if(property_name.empty())
    {
        std::cerr << "Empty property name!" << std::endl;
        return;
    }

    if(property_list.empty())
    {
        std::cerr << "Empty property list!" << std::endl;
        return;
    }

    try
    {
        int size = property_list.size();

        Tango::DevVarStringArray* request = new Tango::DevVarStringArray();
        request->length(size+4);

        (*request)[0] = CORBA::string_dup(device_name.c_str());
        (*request)[1] = CORBA::string_dup("1");
        (*request)[2] = CORBA::string_dup(property_name.c_str());

        std::stringstream size_stream;
        size_stream << size;

        (*request)[3] = CORBA::string_dup(size_stream.str().c_str());

        for(int i=0; i<size; ++i)
            (*request)[i+4] = CORBA::string_dup(property_list.at(i).c_str());

        Tango::DeviceData argin;
        argin << request;

        device_proxy_p->command_inout("DbPutDeviceProperty", argin);
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::list_class_prop()
//==============================================================================
void proxy::list_class_prop(std::string class_name, bool display_value)
{
    if(class_name.empty())
    {
        std::cerr << "Empty class name!" << std::endl;
        return;
    }

    try
    {
        std::vector<std::string> properties =
            db_get_class_property_list(class_name);

        if(display_value)
        {
            std::cout << "Number\tProperty - Value" << std::endl;

            for(unsigned int i=0; i<properties.size(); ++i)
            {
                std::cout << "[" << i << "]\t" << properties.at(i) << " - ";

                std::vector<std::string> values =
                    db_get_class_property(class_name, properties.at(i));

                for(unsigned j=0; j<values.size(); ++j)
                    std::cout << "[" << values.at(j) << "]";

                std::cout << std::endl;
            }
        }
        else
        {
            std::cout << "Number\tProperty" << std::endl;

            for(unsigned int i=0; i<properties.size(); ++i)
            {
                std::cout << "[" << i << "]\t" << properties.at(i)
                    << std::endl;
            }
        }
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::get_class_prop()
//==============================================================================
void proxy::get_class_prop(std::string class_name, std::string property_name)
{
    if(class_name.empty())
    {
        std::cerr << "Empty class name!" << std::endl;
        return;
    }

    if(property_name.empty())
    {
        std::cerr << "Empty property name!" << std::endl;
        return;
    }

    try
    {
        std::cout << "Number\tProperty - Value" << std::endl;

        std::cout << "[0]\t" << property_name << " - ";

        std::vector<std::string> values =
            db_get_class_property(class_name, property_name);

        for(unsigned j=0; j<values.size(); ++j)
            std::cout << "[" << values.at(j) << "]";

        std::cout << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::rm_class_prop()
//==============================================================================
void proxy::rm_class_prop(std::string class_name, std::string property_name)
{
    if(class_name.empty())
    {
        std::cerr << "Empty class name!" << std::endl;
        return;
    }

    if(property_name.empty())
    {
        std::cerr << "Empty property name!" << std::endl;
        return;
    }

    try
    {
        Tango::DevVarStringArray* request = new Tango::DevVarStringArray();
        request->length(2);

        (*request)[0] = CORBA::string_dup(class_name.c_str());
        (*request)[1] = CORBA::string_dup(property_name.c_str());

        Tango::DeviceData argin;
        argin << request;

        device_proxy_p->command_inout("DbDeleteClassProperty", argin);
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::save_class_prop()
//==============================================================================
void proxy::save_class_prop(std::string class_name, std::string property_name,
    std::vector<std::string> property_list)
{
    if(class_name.empty())
    {
        std::cerr << "Empty class name!" << std::endl;
        return;
    }

    if(property_name.empty())
    {
        std::cerr << "Empty property name!" << std::endl;
        return;
    }

    if(property_list.empty())
    {
        std::cerr << "Empty property list!" << std::endl;
        return;
    }

    try
    {
        int size = property_list.size();

        Tango::DevVarStringArray* request = new Tango::DevVarStringArray();
        request->length(size+4);

        (*request)[0] = CORBA::string_dup(class_name.c_str());
        (*request)[1] = CORBA::string_dup("1");
        (*request)[2] = CORBA::string_dup(property_name.c_str());

        std::stringstream size_stream;
        size_stream << size;

        (*request)[3] = CORBA::string_dup(size_stream.str().c_str());

        for(int i=0; i<size; ++i)
            (*request)[i+4] = CORBA::string_dup(property_list.at(i).c_str());

        Tango::DeviceData argin;
        argin << request;

        device_proxy_p->command_inout("DbPutClassProperty", argin);
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::db_get_server_list()
//==============================================================================
std::vector<std::string> proxy::db_get_server_list()
    throw(Tango::DevFailed)
{
    std::string filter ="*";

    Tango::DeviceData argin;
    argin << filter;

    Tango::DeviceData argout =
        device_proxy_p->command_inout("DbGetServerList", argin);

    std::vector<std::string> servers;

    if(!argout.is_empty())
    {
        const Tango::DevVarStringArray* prop_string_array;
        argout >> prop_string_array;

        if(prop_string_array->length() != 0)
        {
            for(unsigned int i=0; i<prop_string_array->length(); ++i)
                servers.push_back((*prop_string_array)[i].in());
        }
        else
            Tango::Except::throw_exception("", "Empty servers list",
                "db_get_server_list");
    }
    else
        Tango::Except::throw_exception("", "Error retrieving data",
            "db_get_server_list");

    return servers;
}

//==============================================================================
//  proxy::db_get_device_class_list()
//==============================================================================
std::vector<std::string> proxy::db_get_device_class_list(std::string server_name)
    throw(Tango::DevFailed)
{
    Tango::DeviceData argin;
    argin << server_name;

    Tango::DeviceData argout =
        device_proxy_p->command_inout("DbGetDeviceClassList", argin);

    std::vector<std::string> device_class_list;

    if(!argout.is_empty())
    {
        const Tango::DevVarStringArray* prop_string_array;
        argout >> prop_string_array;

        if(prop_string_array->length() != 0)
        {
            for(unsigned int i=0; i<prop_string_array->length(); ++i)
                device_class_list.push_back((*prop_string_array)[i].in());
        }
        else
            Tango::Except::throw_exception("", "Server not found",
                "db_get_device_class_list");
    }
    else
        Tango::Except::throw_exception("", "Error retrieving data",
            "db_get_device_class_list");

    return device_class_list;
}

//==============================================================================
//  proxy::db_get_device_property_list()
//==============================================================================
std::vector<std::string> proxy::db_get_device_property_list(std::string device_name)
    throw(Tango::DevFailed)
{
    Tango::DevVarStringArray* request = new Tango::DevVarStringArray();
    request->length(2);

    (*request)[0] = CORBA::string_dup(device_name.c_str());
    (*request)[1] = CORBA::string_dup("*");

    Tango::DeviceData argin;
    argin << request;

    Tango::DeviceData argout =
        device_proxy_p->command_inout("DbGetDevicePropertyList", argin);

    std::vector<std::string> properties;

    if(!argout.is_empty())
    {
        const Tango::DevVarStringArray* prop_string_array;
        argout >> prop_string_array;

        if(prop_string_array->length() != 0)
        {
            for(unsigned int i=0; i<prop_string_array->length(); ++i)
                properties.push_back((*prop_string_array)[i].in());
        }
        else
            Tango::Except::throw_exception("", "Property not found",
                "db_get_device_property_list");
    }
    else
        Tango::Except::throw_exception("", "Error retrieving data",
            "db_get_device_property_list");

    return properties;
}

//==============================================================================
//  proxy::db_get_class_property_list()
//==============================================================================
std::vector<std::string> proxy::db_get_class_property_list(std::string class_name)
    throw(Tango::DevFailed)
{
    Tango::DevString request = CORBA::string_dup(class_name.c_str());

    Tango::DeviceData argin;
    argin << request;

    Tango::DeviceData argout =
        device_proxy_p->command_inout("DbGetClassPropertyList", argin);

    std::vector<std::string> properties;

    if(!argout.is_empty())
    {
        const Tango::DevVarStringArray* prop_string_array;
        argout >> prop_string_array;

        if(prop_string_array->length() != 0)
        {
            for(unsigned int i=0; i<prop_string_array->length(); ++i)
                properties.push_back((*prop_string_array)[i].in());
        }
        else
            Tango::Except::throw_exception("", "Property not found",
                "db_get_class_property_list");
    }
    else
        Tango::Except::throw_exception("", "Error retrieving data",
            "db_get_class_property_list");

    return properties;
}

//==============================================================================
//  proxy::db_get_device_property()
//==============================================================================
std::vector<std::string> proxy::db_get_device_property(std::string device_name,
    std::string property_name) throw(Tango::DevFailed)
{
    Tango::DevVarStringArray* request = new Tango::DevVarStringArray();
    request->length(2);

    (*request)[0] = CORBA::string_dup(device_name.c_str());
    (*request)[1] = CORBA::string_dup(property_name.c_str());

    Tango::DeviceData argin;
    argin << request;

    Tango::DeviceData argout =
        device_proxy_p->command_inout("DbGetDeviceProperty", argin);

    std::vector<std::string> values;

    if(!argout.is_empty())
    {
        const Tango::DevVarStringArray* properties;
        argout >> properties;

        if(properties->length() > 4)
        {
            int size = atoi((*properties)[3]);

            if(size > 0)
            {
                for(int i=4; i<(size+4); ++i)
                    values.push_back((*properties)[i].in());
            }
            else
                Tango::Except::throw_exception("", "Property not found",
                    "db_get_device_property");
        }
        else
            Tango::Except::throw_exception("", "Error on data length",
                "db_get_device_property");
    }
    else
        Tango::Except::throw_exception("", "Error retrieving data",
            "db_get_device_property");

    return values;
}

//==============================================================================
//  proxy::db_get_class_property()
//==============================================================================
std::vector<std::string> proxy::db_get_class_property(std::string class_name,
    std::string property_name) throw(Tango::DevFailed)
{
    Tango::DevVarStringArray* request = new Tango::DevVarStringArray();
    request->length(2);

    (*request)[0] = CORBA::string_dup(class_name.c_str());
    (*request)[1] = CORBA::string_dup(property_name.c_str());

    Tango::DeviceData argin;
    argin << request;

    Tango::DeviceData argout =
        device_proxy_p->command_inout("DbGetClassProperty", argin);

    std::vector<std::string> values;

    if(!argout.is_empty())
    {
        const Tango::DevVarStringArray* properties;
        argout >> properties;

        if(properties->length() > 4)
        {
            int size = atoi((*properties)[3]);

            if(size > 0)
            {
                for(int i=4; i<(size+4); ++i)
                    values.push_back((*properties)[i].in());
            }
            else
                Tango::Except::throw_exception("", "Property not found",
                    "db_get_class_property");
        }
        else
            Tango::Except::throw_exception("", "Error on data length",
                "db_get_class_property");
    }
    else
        Tango::Except::throw_exception("", "Error retrieving data",
            "db_get_class_property");

    return values;
}

}   //namespace