#include <tango.h>
#include <iomanip>
#include <cstring>
#include "proxy.hpp"

//==============================================================================
//Print menu
//==============================================================================
void print_help()
{
    std::cout << "JiveCLI - Tango Jive Command LIne - v0.1"
        << std::endl << std::endl;

    std::cout << "Usage: jivecli {command} [arguments]"
        << std::endl << std::endl;

    std::cout << "Commands:" << std::endl;

    std::cout << " --help -> print this message" << std::endl << std::endl;

    std::cout << "Server and device:" << std::endl;

    std::cout << " --list_servers -> {print_details} -> "
        "print servers list" << std::endl;

    std::cout << " --list_devices -> {server} -> "
        "print devices list for a specified server_name" << std::endl;

    std::cout << " --rm_server -> {server} -> "
        "delete specified server" << std::endl;

    std::cout << " --rm_device -> {device} -> "
        "delete specified device" << std::endl;

    std::cout << " --save_server -> {server} {device} {class} -> "
        "save new server" << std::endl;

    std::cout << " --save_device -> {server} {device} {class} -> "
        "save new device" << std::endl << std::endl;

    std::cout << "Device properties:" << std::endl;

    std::cout << " --list_dev_prop {device} {print_value} -> "
        "print device properties list for a specified device" << std::endl;

    std::cout << " --get_dev_prop {device} {property} -> "
        "print specified device property value" << std::endl;

    std::cout << " --rm_dev_prop {device} {property} -> "
        "remove specified device property" << std::endl;

    std::cout << " --save_dev_prop {device} {property} {values} -> "
        "save property value for a specified device" << std::endl << std::endl;

    std::cout << "Class properties:" << std::endl;

    std::cout << " --list_class_prop {class} {print_value} -> "
        "print class properties list for a specified class" << std::endl;

    std::cout << " --get_class_prop {class} {property} -> "
        "print specified class property value" << std::endl;

    std::cout << " --rm_class_prop {class} {property} -> "
        "remove specified class property" << std::endl;

    std::cout << " --save_class_prop {class} {property} {values} -> "
        "save property value for a specified class" << std::endl << std::endl;

    std::cout << "Arguments:" << std::endl;

    std::cout << " server -> server name in server/instance form" << std::endl;

    std::cout << " device -> device name in xxx/yyy/zzz form" << std::endl;

    std::cout << " class -> class name" << std::endl;

    std::cout << " property -> property name" << std::endl;

    std::cout << " print_details -> print server details" << std::endl
        << "\t 0 -> do not print server details" << std::endl
        << "\t 1 -> print server details" << std::endl;

    std::cout << " print_value -> print property value" << std::endl
        << "\t 0 -> do not print property value" << std::endl
        << "\t 1 -> print property value" << std::endl;

    std::cout << " values -> property values, enclosed in square brackets"
        << std::endl << "\t es: [prop1][prop2]..." << std::endl << std::endl;
}

//==============================================================================
//String split method
//==============================================================================
std::vector<std::string> split_args(std::string argument)
{
    std::vector<std::string> property_list;

    std::size_t begin = 0;
    std::size_t end = 0;

    while( (begin=argument.find_first_of("[")) != std::string::npos &&
        (end=argument.find_first_of("]")) != std::string::npos )
    {
        std::string value = argument.substr(begin+1, end-begin-1);

        property_list.push_back(value);

        argument.erase(begin, end-begin+1);
    }

    return property_list;
}

//==============================================================================
//Main code
//==============================================================================
int main(int argc, char* argv[])
{
    if(argc < 2)
    {
        std::cerr << "jivecli: missing command operand" << std::endl
            << "Try \'jivecli --help\' for more information" << std::endl;
        return 1;
    }

    jive_cli::proxy p;

    std::string command = argv[1];

    if(command.compare("--help")==0)
    {
        print_help();
    }
    else if(command.compare("--list_servers")==0)
    {
        if(argc < 3)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        if(strcmp(argv[2], "0") != 0 && strcmp(argv[2], "1") != 0)
        {
            std::cerr << "astorcli: wrong argument operand" << std::endl
                << "Try \'astorcli --help\' for more information" << std::endl;
            return 1;
        }

        bool display_details;
        std::istringstream(argv[2]) >> display_details;

        p.list_servers(display_details);
    }
    else if(command.compare("--list_devices")==0)
    {
        if(argc < 3)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string server_name = argv[2];

        p.list_devices(server_name);
    }
    else if(command.compare("--rm_server")==0)
    {
        if(argc < 3)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string server_name = argv[2];

        p.rm_server(server_name);
    }
    else if(command.compare("--rm_device")==0)
    {
        if(argc < 3)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string device_name = argv[2];

        p.rm_device(device_name);
    }
    else if(command.compare("--save_server")==0)
    {
        if(argc < 5)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string server_name = argv[2];
        std::string device_name = argv[3];
        std::string class_name = argv[4];

        p.save_server(server_name, device_name, class_name);
    }
    else if(command.compare("--save_device")==0)
    {
        if(argc < 5)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string server_name = argv[2];
        std::string device_name = argv[3];
        std::string class_name = argv[4];

        p.save_device(server_name, device_name, class_name);
    }
    else if(command.compare("--list_dev_prop")==0)
    {
        if(argc < 4)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string device_name = argv[2];

        if(strcmp(argv[3], "0") != 0 && strcmp(argv[3], "1") != 0)
        {
            std::cerr << "astorcli: wrong argument operand" << std::endl
                << "Try \'astorcli --help\' for more information" << std::endl;
            return 1;
        }

        bool display_value;
        std::istringstream(argv[3]) >> display_value;

        p.list_dev_prop(device_name, display_value);
    }
    else if(command.compare("--get_dev_prop")==0)
    {
        if(argc < 4)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string device_name = argv[2];
        std::string property_name = argv[3];

        p.get_dev_prop(device_name, property_name);
    }
    else if(command.compare("--rm_dev_prop")==0)
    {
        if(argc < 4)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string device_name = argv[2];
        std::string property_name = argv[3];

        p.rm_dev_prop(device_name, property_name);
    }
    else if(command.compare("--save_dev_prop")==0)
    {
        if(argc < 5)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string device_name = argv[2];
        std::string property_name = argv[3];
        std::vector<std::string> property_list = split_args(argv[4]);

        p.save_dev_prop(device_name, property_name, property_list);
    }
    else if(command.compare("--list_class_prop")==0)
    {
        if(argc < 4)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string class_name = argv[2];

        if(strcmp(argv[3], "0") != 0 && strcmp(argv[3], "1") != 0)
        {
            std::cerr << "astorcli: wrong argument operand" << std::endl
                << "Try \'astorcli --help\' for more information" << std::endl;
            return 1;
        }

        bool display_value;
        std::istringstream(argv[3]) >> display_value;

        p.list_class_prop(class_name, display_value);
    }
    else if(command.compare("--get_class_prop")==0)
    {
        if(argc < 4)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string class_name = argv[2];
        std::string property_name = argv[3];

        p.get_class_prop(class_name, property_name);
    }
    else if(command.compare("--rm_class_prop")==0)
    {
        if(argc < 4)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string class_name = argv[2];
        std::string property_name = argv[3];

        p.rm_class_prop(class_name, property_name);
    }
    else if(command.compare("--save_class_prop")==0)
    {
        if(argc < 5)
        {
            std::cerr << "jivecli: missing argument operand" << std::endl
                << "Try \'jivecli --help\' for more information" << std::endl;
            return 1;
        }

        std::string class_name = argv[2];
        std::string property_name = argv[3];
        std::vector<std::string> property_list = split_args(argv[4]);

        p.save_class_prop(class_name, property_name, property_list);
    }
    else
    {
        std::cerr << "jivecli: wrong command operand" << std::endl
            << "Try \'jivecli --help\' for more information" << std::endl;
        return 1;
    }

    return 0;
}
