#include <proxy.hpp>
#include <iomanip>

namespace astor_cli
{
    
//==============================================================================
//  proxy::get_host_list()
//==============================================================================
void proxy::get_host_list()
{
    try
    {
        device_proxy_p = new Tango::DeviceProxy(databased_name);

        std::string filter = "*";
        
        Tango::DeviceData argin;
        argin << filter;

        Tango::DeviceData argout = 
            device_proxy_p->command_inout("DbGetHostList", argin);
        
        if(!argout.is_empty())
        {
            const Tango::DevVarStringArray* hosts;
            argout >> hosts;
            
            if(hosts->length() != 0)
            {
                std::cout << "Number\tHost" << std::endl;

                for(unsigned int i=0; i<hosts->length(); ++i)
                    std::cout << "[" << i << "]\t" << (*hosts)[i] << std::endl;
            }
            else
                std::cout << "No host found" << std::endl;
        }
        else
            std::cerr << "Error retrieving data" << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::get_host_servers_info()
//==============================================================================
void proxy::get_host_servers_info(std::string host_name)
{
    try
    {
        device_proxy_p = new Tango::DeviceProxy(databased_name);
        
        Tango::DeviceData argin;
        argin << host_name;

        Tango::DeviceData argout = 
            device_proxy_p->command_inout("DbGetHostServersInfo", argin);
        
        if(!argout.is_empty())
        {
            const Tango::DevVarStringArray* servers_info;
            argout >> servers_info;
            
            if(servers_info->length() != 0)
            {
                int count = 0;
                
                std::cout << "Number\t" << std::left
                    << std::setw(40) << "Server"                    
                    << std::setw(11) << "Controlled"
                    << std::setw(6) << "Level" << std::endl;

                for(unsigned int i=0; i<servers_info->length(); i=i+3)
                {
                    std::cout << "[" << count++ << "]\t" << std::left
                        << std::setw(40) << (*servers_info)[i]
                        << std::setw(11) << (*servers_info)[i+1]
                        << std::setw(6) << (*servers_info)[i+2] << std::endl;
                }
            }
            else
                std::cout << "No server found" << std::endl;
        }
        else
            std::cerr << "Error retrieving data" << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::get_host_running_servers())
//==============================================================================
void proxy::get_host_running_servers(std::string host_name, bool controlled)
{    
    starter_name.append(host_name);
    
    try
    {
        device_proxy_p = new Tango::DeviceProxy(starter_name);
        
        Tango::DeviceData argin;
        argin << controlled;

        Tango::DeviceData argout = 
            device_proxy_p->command_inout("DevGetRunningServers", argin);
        
        if(!argout.is_empty())
        {
            const Tango::DevVarStringArray* servers;
            argout >> servers;
            
            if(servers->length() != 0)
            {
                std::cout << "Number\tServers" << std::endl;

                for(unsigned int i=0; i<servers->length(); ++i)
                    std::cout << "[" << i << "]\t" << (*servers)[i] << std::endl;
            }
            else
                std::cout << "No server found" << std::endl;
        }
        else
            std::cerr << "Error retrieving data" << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::get_host_stopped_servers())
//==============================================================================
void proxy::get_host_stopped_servers(std::string host_name, bool controlled)
{    
    starter_name.append(host_name);
    
    try
    {
        device_proxy_p = new Tango::DeviceProxy(starter_name);
        
        Tango::DeviceData argin;
        argin << controlled;

        Tango::DeviceData argout = 
            device_proxy_p->command_inout("DevGetStopServers", argin);
        
        if(!argout.is_empty())
        {
            const Tango::DevVarStringArray* servers;
            argout >> servers;
            
            if(servers->length() != 0)
            {
                std::cout << "Number\tServers" << std::endl;

                for(unsigned int i=0; i<servers->length(); ++i)
                    std::cout << "[" << i << "]\t" << (*servers)[i] << std::endl;
            }
            else
                std::cout << "No server found" << std::endl;
        }
        else
            std::cerr << "Error retrieving data" << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::start_host_server()
//==============================================================================
void proxy::start_host_server(std::string host_name, std::string server_name)
{    
    starter_name.append(host_name);
    
    try
    {
        device_proxy_p = new Tango::DeviceProxy(starter_name);
        
        Tango::DeviceData argin;
        argin << server_name;

        device_proxy_p->command_inout("DevStart", argin);  
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::stop_host_server()
//==============================================================================
void proxy::stop_host_server(std::string host_name, std::string server_name)
{    
    starter_name.append(host_name);
    
    try
    {
        device_proxy_p = new Tango::DeviceProxy(starter_name);
        
        Tango::DeviceData argin;
        argin << server_name;

        device_proxy_p->command_inout("DevStop", argin);        
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::kill_host_server()
//==============================================================================
void proxy::kill_host_server(std::string host_name, std::string server_name)
{    
    starter_name.append(host_name);
    
    try
    {
        device_proxy_p = new Tango::DeviceProxy(starter_name);
        
        Tango::DeviceData argin;
        argin << server_name;

        device_proxy_p->command_inout("HardKillServer", argin);        
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::get_server_info()
//==============================================================================
void proxy::get_server_info(std::string server_name)
{    
    try
    {
        device_proxy_p = new Tango::DeviceProxy(databased_name);
        
        Tango::DeviceData argin;
        argin << server_name;

        Tango::DeviceData argout = 
            device_proxy_p->command_inout("DbGetServerInfo", argin);
        
        if(!argout.is_empty())
        {
            const Tango::DevVarStringArray* servers_info;
            argout >> servers_info;
            
            int count = 0;

            std::cout << "Number\t" << std::left
                << std::setw(40) << "Server"
                << std::setw(20) << "Host"    
                << std::setw(11) << "Controlled"
                << std::setw(6) << "Level" << std::endl;
            
            if(servers_info->length()==4)
            {
                    std::cout << "[" << count++ << "]\t" << std::left
                        << std::setw(40) << (*servers_info)[0]
                        << std::setw(20) << (*servers_info)[1]    
                        << std::setw(11) << (*servers_info)[2]
                        << std::setw(6) << (*servers_info)[3] << std::endl;                    
            }
            else
                std::cerr << "Not enough info retrieved" << std::endl;
        }
        else
            std::cerr << "Error retrieving data" << std::endl;
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

//==============================================================================
//  proxy::put_server_info()
//==============================================================================
void proxy::put_server_info(std::string server_name, std::string host_name,
    unsigned int controlled, unsigned int startup)
{          
    if(server_name.empty())
    {
        std::cerr << "Empty server name!" << std::endl;
        return;
    }
   
    if(host_name.empty())
    {
        std::cerr << "Empty host name!" << std::endl;
        return;
    }
    
    if(controlled > 1)
    {
        std::cerr << "Invalid controlled value, valid [0,1]" << std::endl;
        return;
    }

    if(startup > 5)
    {
        std::cerr << "Invalid startup level value, valid [0-5]" << std::endl;
        return;
    }

    Tango::DevVarStringArray* servers_info = new Tango::DevVarStringArray();
    servers_info->length(4);

    (*servers_info)[0] = CORBA::string_dup(server_name.c_str());
    (*servers_info)[1] = CORBA::string_dup(host_name.c_str());  
    (*servers_info)[2] = CORBA::string_dup(int_to_string(controlled).c_str());
    (*servers_info)[3] = CORBA::string_dup(int_to_string(startup).c_str());
    
    try
    {
        device_proxy_p = new Tango::DeviceProxy(databased_name);
      
        Tango::DeviceData argin;
        argin << servers_info;        
        
        device_proxy_p->command_inout("DbPutServerInfo", argin);
    }
    catch(Tango::DevFailed& ex)
    {
        print_exception(ex);
    }
}

}   //namespace