<?php

/*
 * This file is part of rap
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

/**
 * Data model for the user. An user is a set of identities.
 */
class User {

    // User ID
    public $id;
    // List of identities
    public $identities;

    public function __construct() {
        $this->identities = [];
    }

    public function addIdentity(Identity $identity): void {
        array_push($this->identities, $identity);
    }

    public function getIdentityByTypedId(string $typedId): Identity {
        foreach ($this->identities as $identity) {
            if ($identity->typedId === $typedId) {
                return $identity;
            }
        }
        throw new \Exception("Identity not found for typed id " . $typedId);
    }

    public function getPrimaryEmail() {
        foreach ($this->identities as $identity) {
            if ($identity->primary) {
                return $identity->email;
            }
        }
        // A primary identity MUST be defined
        throw new \Exception("No primary identity defined for user " . $this->id);
    }

    /**
     * Returns name and surname if they are present, preferring the primary identity data.
     */
    public function getCompleteName(): ?string {

        $completeName = null;

        foreach ($this->identities as $identity) {
            if ($identity->name !== null && $identity->surname !== null) {
                $completeName = $identity->name . ' ' . $identity->surname;
            }
            if ($identity->primary && $completeName !== null) {
                break;
            }
        }

        return $completeName;
    }

    /**
     * Returns the user name if it is present, preferring the primary identity data.
     */
    public function getName(): ?string {

        $name = null;

        foreach ($this->identities as $identity) {
            if ($identity->name !== null) {
                $name = $identity->name;
            }
            if ($identity->primary && $name !== null) {
                break;
            }
        }

        return $name;
    }

    /**
     * Returns the user surname if it is present, preferring the primary identity data.
     */
    public function getSurname(): ?string {

        $surname = null;

        foreach ($this->identities as $identity) {
            if ($identity->surname !== null) {
                $surname = $identity->surname;
            }
            if ($identity->primary && $surname !== null) {
                break;
            }
        }

        return $surname;
    }

    /**
     * Returns the user institution if it is present, preferring the primary identity data.
     */
    public function getInstitution(): ?string {

        $institution = null;

        foreach ($this->identities as $identity) {
            if ($identity->institution !== null) {
                $institution = $identity->institution;
            }
            if ($identity->primary && $institution !== null) {
                break;
            }
        }

        return $institution;
    }

}
