<?php

/*
 * This file is part of rap
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

/**
 * This class wraps various objects that need to be stored into the session in
 * order to provide an object oriented transparent session management.
 */
class SessionData {

    const KEY = "SessionData";

    private $user;
    private $userToJoin;
    private $x509DataToRegister;
    private $oauth2RequestData;
    private $action;
    private $loginIdentityType;
    private $autojoin = false;
    private $rejectedJoins = [];

    public function setUser(?User $user): void {
        $this->user = $user;
        $this->save();
    }

    public function getUser(): ?User {
        return $this->user;
    }

    public function setUserToJoin(?User $userToJoin): void {
        $this->userToJoin = $userToJoin;
        $this->save();
    }

    public function getUserToJoin(): ?User {
        return $this->userToJoin;
    }

    public function setAutojoin(bool $autojoin): void {
        $this->autojoin = $autojoin;
        $this->save();
    }

    public function isAutojoin(): bool {
        return $this->autojoin;
    }

    public function addRejectedJoin(string $userId): void {
        array_push($this->rejectedJoins, $userId);
        $this->save();
    }

    public function getRejectedJoins(): array {
        return $this->rejectedJoins;
    }

    /**
     * Used for logging the identity type chosen for the login at the end of the login process
     */
    public function setLoginIdentityType(string $loginIdentityType): void {
        $this->loginIdentityType = $loginIdentityType;
        $this->save();
    }

    public function getLoginIdentityType(): ?string {
        return $this->loginIdentityType;
    }

    /**
     * Update the user data model stored into the session after the primary
     * identity has changed, in order to avoid reading again the user data from
     * the database.
     * @param int $identityId
     */
    public function updatePrimaryIdentity($identityId): void {
        foreach ($this->user->identities as $identity) {
            $identity->primary = ($identity->id === $identityId);
        }
        $this->save();
    }

    public function setX509DataToRegister(?X509Data $x509DataToRegister): void {
        $this->x509DataToRegister = $x509DataToRegister;
        $this->save();
    }

    public function getX509DataToRegister(): ?X509Data {
        return $this->x509DataToRegister;
    }

    public function setOAuth2RequestData(?OAuth2RequestData $oauth2RequestData): void {
        $this->oauth2RequestData = $oauth2RequestData;
        $this->save();
    }

    public function getOAuth2RequestData(): ?OAuth2RequestData {
        return $this->oauth2RequestData;
    }

    public function setAction(?string $action): void {
        $this->action = $action;
        $this->save();
    }

    public function getAction(): ?string {
        return $this->action;
    }

    /**
     * Store the data into the $_SESSION PHP variable
     */
    public function save() {
        $_SESSION[SessionData::KEY] = $this;
    }

}
