<?php

/*
 * This file is part of rap
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

/**
 * RFC 6749 specify that in some situations the client must send an Authorization
 * Basic header containing its credentials (access token in the authorization code
 * flow and refresh token requests).
 */
class ClientAuthChecker {

    private $locator;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
    }

    public function validateClientAuth(array $headers): BrowserBasedOAuth2Client {

        $basic = $this->getBasicAuthArray($headers);

        $clientId = $basic[0];
        $clientSecret = $basic[1];

        return $this->locator->getBrowserBasedOAuth2ClientByIdAndSecret($clientId, $clientSecret);
    }

    public function validateCliClientAuth(array $headers): CliOAuth2Client {

        $basic = $this->getBasicAuthArray($headers);

        $clientId = $basic[0];
        $clientSecret = $basic[1];

        return $this->locator->getCliClientByIdAndSecret($clientId, $clientSecret);
    }

    private function getBasicAuthArray($headers): array {

        if (!isset($headers['Authorization'])) {
            throw new UnauthorizedException("Missing Authorization header");
        }

        $authorizationHeader = explode(" ", $headers['Authorization']);
        if ($authorizationHeader[0] === "Basic") {
            $basic = explode(':', base64_decode($authorizationHeader[1]));
            if (count($basic) !== 2) {
                throw new BadRequestException("Malformed Basic-Auth header");
            }
            return $basic;
        } else {
            throw new UnauthorizedException("Expected Basic authorization header");
        }
    }

}
