<?php

/*
 * This file is part of rap
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

/**
 * Perform operations on users.
 */
class UserHandler {

    private $locator;
    private $userDAO;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
        $this->userDAO = $locator->getUserDAO();
    }

    /**
     * Update user information into the database, creating a new user or adding
     * new identities to it.
     * @param \RAP\User $user
     */
    public function saveUser(User $user): void {

        $primarySpecified = true;

        // If new user
        if ($user->id === null) {
            $primarySpecified = false;
            $user->id = $this->userDAO->createUser();
        }

        foreach ($user->identities as $identity) {
            if ($identity->id === null) {
                $identity->id = $this->userDAO->insertIdentity($identity, $user->id);
                if (!$primarySpecified) {
                    $this->userDAO->setPrimaryIdentity($user->id, $identity->id);
                    $identity->primary = true;
                }
            }
        }
    }

    /**
     * Update user with fresh information received by IdP. Useful for keeping email address always updated.
     */
    public function updateIdentity(User $user, Identity $identity): void {

        $savedIdentity = $user->getIdentityByTypedId($identity->typedId);
        $savedIdentity->email = $identity->email;
        if ($identity->name !== null) {
            $savedIdentity->name = $identity->name;
        }
        if ($identity->surname !== null) {
            $savedIdentity->surname = $identity->surname;
        }
        if ($identity->institution !== null) {
            $savedIdentity->institution = $identity->institution;
        }
        $this->userDAO->updateIdentity($savedIdentity);
    }

    public function joinUsers(User $user1, User $user2): User {

        $userId1 = $user1->id;
        $userId2 = $user2->id;

        if ($userId1 === $userId2) {
            return $user1;
        }

        if ($userId1 === null && $userId2 === null) {
            throw new BadRequestException("Called join on two users that are both new");
        }

        if ($userId1 === null) {
            return $this->joinNewIdentity($user2, $user1);
        }
        if ($userId2 === null) {
            return $this->joinNewIdentity($user1, $user2);
        }

        // Call GMS for moving groups and privileges from one user to the other
        $remainingUserId = $this->locator->getGmsClient()->joinGroups($userId1, $userId2);

        $remainingUser = $userId1 === $remainingUserId ? $user1 : $user2;
        $userToDelete = $userId1 === $remainingUserId ? $user2 : $user1;

        // Call DAO for performing join operation into the RAP database.
        $this->userDAO->joinUsers($remainingUserId, $userToDelete->id);

        $this->moveIdentities($remainingUser, $userToDelete);

        // merged user
        return $remainingUser;
    }

    /**
     * New identity, not yet associated with a user: simply add it to the other user.
     */
    private function joinNewIdentity(User $user, User $newUser): User {
        $identity = $newUser->identities[0];
        $user->addIdentity($identity);
        $this->saveUser($user);
        return $user;
    }

    private function moveIdentities(User $remainingUser, User $deletedUser) {
        foreach ($deletedUser->identities as $identity) {
            $identity->primary = false;
            $remainingUser->addIdentity($identity);
        }
    }

}
