<?php

/*
 * This file is part of rap
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

class MySQLAccessTokenDAO extends BaseMySQLDAO implements AccessTokenDAO {

    public function __construct(Locator $locator) {
        parent::__construct($locator);
    }

    public function createTokenData(AccessTokenData $tokenData): AccessTokenData {

        $dbh = $this->getDBHandler();
        $stmt = $dbh->prepare("INSERT INTO access_token (code_hash, user_id, redirect_uri, client_id, scope, creation_time, expiration_time)"
                . " VALUES(:code_hash, :user_id, :redirect_uri, :client_id, :scope, :creation_time, :expiration_time)");

        $scope = null;
        if ($tokenData->scope !== null) {
            $scope = join(' ', $tokenData->scope);
        }

        $params = array(
            ':code_hash' => $tokenData->codeHash,
            ':user_id' => $tokenData->userId,
            ':redirect_uri' => $tokenData->redirectUri,
            ':client_id' => $tokenData->clientId,
            ':scope' => $scope,
            ':creation_time' => $tokenData->creationTime,
            ':expiration_time' => $tokenData->expirationTime
        );

        if ($stmt->execute($params)) {
            return $tokenData;
        } else {
            error_log($stmt->errorInfo()[2]);
            throw new \Exception("SQL error while storing user token");
        }
    }

    public function retrieveTokenDataFromCode(string $codeHash): ?AccessTokenData {

        $dbh = $this->getDBHandler();

        // Access token can be retrieved from code in 1 minute from the creation
        $stmt = $dbh->prepare("SELECT user_id, redirect_uri, client_id, creation_time, expiration_time, scope "
                . " FROM access_token WHERE code_hash = :code_hash AND UNIX_TIMESTAMP() < (creation_time + 60)");
        $stmt->bindParam(':code_hash', $codeHash);

        $stmt->execute();

        $row = $stmt->fetch();
        if (!$row) {
            return null;
        }

        return $this->getTokenDataFromRow($row);
    }

    private function getTokenDataFromRow(array $row): AccessTokenData {

        $token = new AccessTokenData();
        $token->userId = $row['user_id'];
        $token->redirectUri = $row['redirect_uri'];
        $token->clientId = $row['client_id'];
        $token->creationTime = $row['creation_time'];
        $token->expirationTime = $row['expiration_time'];

        $scope = null;
        if (isset($row['scope'])) {
            $scope = $row['scope'];
        }
        if ($scope !== null && $scope !== '') {
            $token->scope = explode(' ', $scope);
        }

        return $token;
    }

    function deleteTokenData(string $codeHash): void {

        $dbh = $this->getDBHandler();

        $stmt = $dbh->prepare("DELETE FROM access_token WHERE code_hash = :code_hash");
        $stmt->bindParam(':code_hash', $codeHash);

        $stmt->execute();
    }

}
