<?php

/*
 * This file is part of rap
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

interface UserDAO {

    /**
     * Create a new identity.
     * @param type $userId the user ID associated to that identity
     * @return type the new identity ID
     */
    function insertIdentity(Identity $identity, $userId);

    /**
     * Create a new user.
     * @return the new user ID.
     */
    function createUser();

    /**
     * @return RAP\User an user object, null if nothing was found.
     */
    function findUserById(string $userId): ?User;

    function setPrimaryIdentity($userId, $identityId);

    /**
     * Retrieve the user associated to a given identity using the typedId.
     * @param type $type Identity type (EDU_GAIN, X509, GOOGLE, ...)
     * @param type $typedId typed unique value used to search the identity in the database
     * @return RAP\User an user object, null if nothing was found.
     */
    function findUserByIdentity($type, $typedId): ?User;

    /**
     * Retrieve a set of users matching a given search text.
     * @param type $searchText name, surname or email
     * @return list of RAP\User objects
     */
    function searchUser($searchText);

    /**
     * Retrieve a list of all users having given identifiers.
     * @param array $identifiers
     * @return array
     */
    function getUsers(array $identifiers): array;

    /**
     * Perform a join request.
     * @param type $userId1 the user that will receive all identities
     * @param type $userId2 the user that will lost the identities and will be
     * deleted from the database
     */
    function joinUsers($userId1, $userId2);

    function isAdmin($userId): bool;

    function updateIdentity(Identity $identity): void;

    function findJoinableUsersByEmail(string $email): array;

    function findJoinableUsersByUserId(string $userId): array;

    function insertRejectedJoin(string $userId1, string $userId2): void;
}
